/**
 * @license Copyright (c) 2013, CKSource - Frederico Knabben
 * For licensing, see LICENSE
 */

"use strict";

/* global describe: false, it: false */

var assert = require("assert"),
    p = require("path"),
    // ScanningBranch = require(global.MODULE_ROOT + "/libraries/scripts/JsLoader/FileSystem/ScanningBranch"),
    ScanningProcess = require(global.MODULE_ROOT + "/libraries/scripts/JsLoader/FileSystem/ScanningProcess");

describe("JsLoader/FileSystem/ScanningProcess", function () {
    describe("#expectPath", function () {
        it("cannot expect same path twice", function () {
            var expectedPath = p.sep + "foo",
                scanningProcess = new ScanningProcess({});

            scanningProcess.expectPath(expectedPath);

            assert.throws(function () {
                scanningProcess.expectPath(expectedPath);
            });
        });
    });

    describe("#getBranch", function () {
        it("does not create branch automatically if it does not exist", function () {
            var scanningProcess = new ScanningProcess({});

            assert.throws(function () {
                scanningProcess.getBranch(p.sep + "foo");
            });
        });
    });

    describe("#hasBranch", function () {
        it("determines if scannin process has branch", function () {
            var scanningProcess = new ScanningProcess({});

            assert.ok(!scanningProcess.hasBranch(["a"].join(p.sep)));

            scanningProcess.onDirectoryVisited(["a"].join(p.sep));

            assert.ok(scanningProcess.hasBranch(["a"].join(p.sep)));
            assert.ok(scanningProcess.hasBranch(["a", "b"].join(p.sep)));
            assert.ok(scanningProcess.hasBranch(["a", "b", "c"].join(p.sep)));
        });

        it("does not accept files with same name prefix", function () {
            var scanningProcess = new ScanningProcess({});

            assert.ok(!scanningProcess.hasBranch(["a"].join(p.sep)));

            scanningProcess.onFileVisited(["a"].join(p.sep));

            assert.ok(scanningProcess.hasBranch(["a"].join(p.sep)));
            assert.ok(!scanningProcess.hasBranch(["ab"].join(p.sep)));
        });
    });

    describe("#isStopped", function () {
        it("is not stopped when expected paths are pending", function () {
            var scanningProcess = new ScanningProcess({});

            scanningProcess.expectPath(p.sep + "foo");

            assert.ok(!scanningProcess.isStopped());
        });
    });

    describe("#onDirectoryVisited", function () {
        it("creates new branches only if they are not covered", function () {
            var scanningProcess = new ScanningProcess({});

            assert.strictEqual(scanningProcess.branches.length, 0);

            scanningProcess.onDirectoryVisited(["a"].join(p.sep));

            assert.strictEqual(scanningProcess.branches.length, 1);

            scanningProcess.onDirectoryVisited(["a", "b"].join(p.sep));
            scanningProcess.onDirectoryVisited(["a", "b", "c"].join(p.sep));

            assert.strictEqual(scanningProcess.branches.length, 1);

            scanningProcess.onDirectoryVisited(["a", "b", "d"].join(p.sep));

            assert.strictEqual(scanningProcess.branches.length, 2);

            scanningProcess.onDirectoryVisited(["a", "e"].join(p.sep));

            assert.strictEqual(scanningProcess.branches.length, 3);

            scanningProcess.onDirectoryVisited(["a", "e", "f"].join(p.sep));

            assert.strictEqual(scanningProcess.branches.length, 3);
        });
    });

    describe("#onFileVisited", function () {
        it("creates new branches only if they are not covered", function () {
            var scanningProcess = new ScanningProcess({});

            assert.strictEqual(scanningProcess.branches.length, 0);

            scanningProcess.onDirectoryVisited(["a"].join(p.sep));

            assert.strictEqual(scanningProcess.branches.length, 1);

            scanningProcess.onFileVisited(["a", "b"].join(p.sep));

            assert.strictEqual(scanningProcess.branches.length, 1);

            scanningProcess.onFileVisited(["a", "c"].join(p.sep));

            assert.strictEqual(scanningProcess.branches.length, 2);
        });
    });

    describe("#unexpectPath", function () {
        it("makes path not expected to be visited", function () {
            var expectedPath = "/foo",
                scanningProcess = new ScanningProcess({});

            scanningProcess.expectPath(expectedPath);
            assert.ok(scanningProcess.isExpectingPath(expectedPath));
            scanningProcess.unexpectPath(expectedPath);
            assert.ok(!scanningProcess.isExpectingPath(expectedPath));
        });
        it("cannot unexpect not expected path", function () {
            var scanningProcess = new ScanningProcess({});

            assert.throws(function () {
                scanningProcess.unexpectPath(p.sep + "foo");
            });
        });
    });
});
