/**
 * @license Copyright (c) 2013, CKSource - Frederico Knabben
 * For licensing, see LICENSE
 */

"use strict";

/* global describe: false, it: false */

var assert = require("assert"),
    Synchronous = require(global.MODULE_ROOT + "/libraries/scripts/JsLoader/FileSystem/FileSystemStrategy/Synchronous");

describe("JsLoader/FileSystem/FileSystemStrategy/Synchronous", function () {

    describe("#exists", function () {

        it("checks that file exists", function () {
            var exists,
                synchronous = new Synchronous();

            synchronous.exists(__filename, function (internalExists) {
                exists = internalExists;
            });

            assert.ok(exists);
        });

        it("checks that file does not exist", function () {
            var exists,
                inputPath = __filename + __filename,
                synchronous = new Synchronous();

            synchronous.exists(inputPath, function (internalExists) {
                exists = internalExists;
            });

            assert.ok(!exists);
        });

    });

    describe("#hasDirDir", function () {

        it("checks that directory has directory", function () {
            var exists,
                inputDirectory = __dirname + "/../",
                inputPath = "FileSystemStrategy",
                synchronous = new Synchronous();

            synchronous.hasDirDir(inputDirectory, inputPath, function (innerExists, err) {
                exists = innerExists;
            });

            assert.ok(exists);
        });

        it("checks that directory has file that is not a directory", function () {
            var exists,
                inputDirectory = __dirname,
                inputPath = __filename,
                synchronous = new Synchronous();

            synchronous.hasDirDir(inputDirectory, inputPath, function (innerExists, err) {
                exists = innerExists;
            });

            assert.ok(!exists);
        });

        it("checks that directory does not have a directory", function () {
            var exists,
                inputDirectory = __dirname,
                inputPath = "foo",
                synchronous = new Synchronous();

            synchronous.hasDirDir(inputDirectory, inputPath, function (innerExists, err) {
                exists = innerExists;
            });

            assert.ok(!exists);
        });

    });

    describe("#hasDirFile", function () {

        it("checks that directory has file", function () {
            var exists,
                inputDirectory = __dirname,
                inputPath = "AsynchronousTest.js",
                synchronous = new Synchronous();

            synchronous.hasDirFile(inputDirectory, inputPath, function (innerExists, err) {
                exists = innerExists;
            });

            assert.ok(exists);
        });

        it("checks that directory has file that is a directory", function () {
            var exists,
                inputDirectory = __dirname + "/../",
                inputPath = "FileSystemStrategy",
                synchronous = new Synchronous();

            synchronous.hasDirFile(inputDirectory, inputPath, function (innerExists, err) {
                exists = innerExists;
            });

            assert.ok(!exists);
        });

        it("checks that directory does not have a file", function () {
            var exists,
                inputDirectory = __dirname,
                inputPath = "foo",
                synchronous = new Synchronous();

            synchronous.hasDirFile(inputDirectory, inputPath, function (innerExists, err) {
                exists = innerExists;
            });

            assert.ok(!exists);
        });

    });

    describe("#hasDirPath", function () {

        it("returns an error when given path is not a directory", function () {
            var err,
                exists,
                synchronous = new Synchronous();

            synchronous.hasDirPath(__filename, "foo", function (innerExists, innerErr) {
                err = innerErr;
                exists = innerExists;
            });

            assert.ok(err);
            assert.ok(!exists);
        });

    });

    describe("#isDirectory", function () {

        it("responds false on non-existent path", function () {
            var isDirectory,
                synchronous = new Synchronous();

            synchronous.isDirectory(__dirname + "/non-existent-path", function (innerIsDirectory) {
                isDirectory = innerIsDirectory;
            });

            assert.ok(!isDirectory);
        });

    });

    describe("#isFile", function () {

        it("responds false on non-existent path", function () {
            var isFile,
                synchronous = new Synchronous();

            synchronous.isFile(__dirname + "/non-existent-path", function (innerIsFile) {
                isFile = innerIsFile;
            });

            assert.ok(!isFile);
        });

    });

    describe("#realpath", function () {

        it("resolved real file path", function () {
            var err,
                realpath,
                synchronous = new Synchronous();

            synchronous.realpath(__dirname, function (innerRealpath, innerErr) {
                err = innerErr;
                realpath = innerRealpath;
            });

            assert.equal(realpath, __dirname);
            assert.ok(!err);
        });

        it("failes resolving real file path", function () {
            var err,
                realpath,
                synchronous = new Synchronous();

            synchronous.realpath(null, function (innerRealpath, innerErr) {
                err = innerErr;
                realpath = innerRealpath;
            });

            assert.ok(!realpath);
            assert.ok(err);
        });

    });

    describe("#readDir", function () {

        it("returns error on failure", function () {
            var err,
                files,
                synchronous = new Synchronous();

            synchronous.readDir(__dirname + __dirname, function (innerFiles, innerErr) {
                err = innerErr;
                files = innerFiles;
            });

            assert.ok(err);
            assert.ok(!Array.isArray(files));
            assert.ok(!files);
        });

        it("reads all files from a directory", function () {
            var err,
                files,
                synchronous = new Synchronous();

            synchronous.readDir(__dirname, function (innerFiles, innerErr) {
                err = innerErr;
                files = innerFiles;
            });

            assert.ok(!err);
            assert.ok(Array.isArray(files));
        });

    });

    describe("#readDirs", function () {

        it("reads all files from directories", function () {
            var assertion = {},
                files,
                synchronous = new Synchronous();

            assertion[global.MODULE_ROOT_TESTS + "/fixtures/search_site_1/directory_1"] = [
                "search_target.txt",
                "search_tgt_2.txt",
                "search_tgt_3.txt"
            ];
            assertion[global.MODULE_ROOT_TESTS + "/fixtures/search_site_1/directory_2"] = [
                "directory_5"
            ];

            synchronous.readDirs([
                global.MODULE_ROOT_TESTS + "/fixtures/search_site_1/directory_1",
                global.MODULE_ROOT_TESTS + "/fixtures/search_site_1/directory_2"
            ], function (innerFiles, err) {
                assert.ok(!err);
                files = innerFiles;
            });

            assert.deepEqual(files, assertion);
        });

    });

    describe("#readFile", function () {

        it("returns error on failure", function () {
            var contents,
                err,
                synchronous = new Synchronous(),
                inputFilename = "non existent file";

            synchronous.readFile(inputFilename, {
                encoding: "utf8"
            }, function (innerContents, innerErr) {
                contents = innerContents;
                err = innerErr;
            });

            assert.ok(err);
            assert.ok(!contents);
        });

        it("reads this file contents", function () {
            var contents,
                err,
                synchronous = new Synchronous(),
                inputFilename = global.MODULE_ROOT_TESTS + "/fixtures/test_file_1.txt";

            synchronous.readFile(inputFilename, {
                encoding: "utf8"
            }, function (innerContents, innerErr) {
                contents = innerContents;
                err = innerErr;
            });

            assert.equal(contents, "test_file_1_contents\n");
            assert.ok(!err);
        });

    });

    describe("#stat", function () {

        it("returns error on failure", function () {
            var err,
                stats,
                synchronous = new Synchronous(),
                inputFilename = "non existent file";

            synchronous.stat(inputFilename, function (innerStats, innerErr) {
                stats = innerStats;
                err = innerErr;
            });

            assert.ok(err);
            assert.ok(!stats);
        });

    });

    describe("#statPaths", function () {

        it("stats all given paths", function () {
            var stats,
                synchronous = new Synchronous();

            synchronous.statPaths([__dirname, __filename], function (innserStats, err) {
                assert.ok(!err);
                stats = innserStats;
            });

            assert.ok(stats[__dirname].isDirectory());
            assert.ok(stats[__filename].isFile());
        });

    });

});
