/**
 * @license Copyright (c) 2013, CKSource - Frederico Knabben
 * For licensing, see LICENSE
 */

"use strict";

/* global describe: false, it: false */

var assert = require("assert"),
    Asynchronous = require(global.MODULE_ROOT + "/libraries/scripts/JsLoader/FileSystem/FileSystemStrategy/Asynchronous");

describe("JsLoader/FileSystem/FileSystemStrategy/Asynchronous", function () {

    describe("#exists", function () {

        it("checks that file exists", function (done) {
            var asynchronous = new Asynchronous();

            asynchronous.exists(__filename, function (exists) {
                assert.ok(exists);
                done();
            });
        });

        it("checks that file does not exist", function (done) {
            var asynchronous = new Asynchronous();

            asynchronous.exists(__filename, function (exists) {
                assert.ok(exists);
                done();
            });
        });

    });

    describe("#hasDirDir", function () {

        it("checks that directory has directory", function (done) {
            var asynchronous = new Asynchronous(),
                inputDirectory = __dirname + "/../",
                inputPath = "FileSystemStrategy";

            asynchronous.hasDirDir(inputDirectory, inputPath, function (exists, err) {
                assert.ok(exists);
                done();
            });
        });

        it("checks that directory has file that is not a directory", function (done) {
            var asynchronous = new Asynchronous(),
                inputDirectory = __dirname,
                inputPath = __filename;

            asynchronous.hasDirDir(inputDirectory, inputPath, function (exists, err) {
                assert.ok(!exists);
                done();
            });
        });

        it("checks that directory does not have a directory", function (done) {
            var asynchronous = new Asynchronous(),
                inputDirectory = __dirname,
                inputPath = "foo";

            asynchronous.hasDirDir(inputDirectory, inputPath, function (exists, err) {
                assert.ok(!exists);
                done();
            });
        });

    });

    describe("#hasDirFile", function () {

        it("checks that directory has file", function (done) {
            var asynchronous = new Asynchronous(),
                inputDirectory = __dirname,
                inputPath = "AsynchronousTest.js";

            asynchronous.hasDirFile(inputDirectory, inputPath, function (exists, err) {
                assert.ok(exists);
                done();
            });
        });

        it("checks that directory has file that is a directory", function (done) {
            var asynchronous = new Asynchronous(),
                inputDirectory = __dirname + "/../",
                inputPath = "FileSystemStrategy";

            asynchronous.hasDirFile(inputDirectory, inputPath, function (exists, err) {
                assert.ok(!exists);
                done();
            });
        });

        it("checks that directory does not have a file", function (done) {
            var asynchronous = new Asynchronous(),
                inputDirectory = __dirname,
                inputPath = "foo";

            asynchronous.hasDirFile(inputDirectory, inputPath, function (exists, err) {
                assert.ok(!exists);
                done();
            });
        });

    });

    describe("#hasDirPath", function () {

        it("returns an error when given path is not a directory", function (done) {
            var asynchronous = new Asynchronous();

            asynchronous.hasDirPath(__filename, "foo", function (exists, err) {
                assert.ok(err);
                assert.ok(!exists);

                done();
            });
        });

    });

    describe("#realpath", function () {

        it("resolved real file path", function (done) {
            var asynchronous = new Asynchronous();

            asynchronous.realpath(__dirname, function (realpath, err) {
                assert.equal(realpath, __dirname);
                assert.ok(!err);

                done();
            });
        });

    });

    describe("#readDir", function () {

        it("reads all files from a directory", function (done) {
            var asynchronous = new Asynchronous();

            asynchronous.readDir(__dirname, function (files, err) {
                assert.ok(!err);
                assert.ok(Array.isArray(files));

                done();
            });
        });

    });

    describe("#readDirs", function () {

        it("reads all files from directories", function (done) {
            var assertion = {},
                asynchronous = new Asynchronous();

            assertion[global.MODULE_ROOT_TESTS + "/fixtures/search_site_1/directory_1"] = [
                "search_target.txt",
                "search_tgt_2.txt",
                "search_tgt_3.txt"
            ];
            assertion[global.MODULE_ROOT_TESTS + "/fixtures/search_site_1/directory_2"] = [
                "directory_5"
            ];

            asynchronous.readDirs([
                global.MODULE_ROOT_TESTS + "/fixtures/search_site_1/directory_1",
                global.MODULE_ROOT_TESTS + "/fixtures/search_site_1/directory_2"
            ], function (files, err) {
                assert.ok(!err);
                assert.deepEqual(files, assertion);

                done();
            });
        });

    });

    describe("#readFile", function () {

        it("reads this file contents", function (done) {
            var asynchronous = new Asynchronous(),
                inputFilename = global.MODULE_ROOT_TESTS + "/fixtures/test_file_1.txt";

            asynchronous.readFile(inputFilename, {
                encoding: "utf8"
            }, function (contents, err) {
                assert.ok(!err);
                assert.equal(contents, "test_file_1_contents\n");

                done();
            });
        });

    });

    describe("#statPaths", function () {

        it("stats all given paths", function (done) {
            var asynchronous = new Asynchronous();

            asynchronous.statPaths([__dirname, __filename], function (stats, err) {
                assert.ok(!err);
                assert.ok(stats[__dirname].isDirectory());
                assert.ok(stats[__filename].isFile());

                done();
            });
        });

    });

});
