/**
 * @license Copyright (c) 2013, CKSource - Frederico Knabben
 * For licensing, see LICENSE
 */

"use strict";

/* global describe: false, it: false */

var assert = require("assert"),
    p = require("path"),
    AsynchronousFileSystemStrategy = require(global.MODULE_ROOT + "/libraries/scripts/JsLoader/FileSystem/FileSystemStrategy/Asynchronous"),
    DirectoryScanningStrategy = require(global.MODULE_ROOT + "/libraries/scripts/JsLoader/FileSystem/ScanningStrategy/Pattern/Directory"),
    FileScanningStrategy = require(global.MODULE_ROOT + "/libraries/scripts/JsLoader/FileSystem/ScanningStrategy/Pattern/File"),
    FileSystemScanner = require(global.MODULE_ROOT + "/libraries/scripts/JsLoader/FileSystem/FileSystemScanner"),
    FileSystemStrategy = require(global.MODULE_ROOT + "/libraries/scripts/JsLoader/FileSystem/FileSystemStrategy"),
    ScanningStrategy = require(global.MODULE_ROOT + "/libraries/scripts/JsLoader/FileSystem/ScanningStrategy"),
    SynchronousFileSystemStrategy = require(global.MODULE_ROOT + "/libraries/scripts/JsLoader/FileSystem/FileSystemStrategy/Synchronous");

describe("JsLoader/FileSystem/FileSystemScanner", function () {

    describe("in general", function () {

        it("requires ScanningStrategy and FileSystemStrategy to be constructed", function () {
            var fileSystemScanner,
                fileSystemStrategy = new FileSystemStrategy(),
                scanningStrategy = new DirectoryScanningStrategy(/.*/);

            assert.throws(function () {
                fileSystemScanner = new FileSystemScanner();
            }, Error);

            assert.throws(function () {
                fileSystemScanner = new FileSystemScanner("no a scanning strategy");
            }, Error);

            assert.throws(function () {
                fileSystemScanner = new FileSystemScanner("no a scanning strategy", "not a filesystem strategy");
            }, Error);

            assert.throws(function () {
                fileSystemScanner = new FileSystemScanner(scanningStrategy, "not a filesystem strategy");
            }, Error);

            assert.throws(function () {
                fileSystemScanner = new FileSystemScanner("no a scanning strategy", fileSystemStrategy);
            }, Error);
        });

        it("does not accept scanning strategy that looks for nothing", function () {
            var fileSystemScanner,
                fileSystemStrategy = new FileSystemStrategy(),
                scanningStrategy = new ScanningStrategy();

            scanningStrategy.isLookingForDirectory = false;
            scanningStrategy.isLookingForFile = false;

            assert.throws(function () {
                fileSystemScanner = new FileSystemScanner(scanningStrategy, fileSystemStrategy);
            }, Error);
        });

    });

    describe("#scanDirectory", function () {

        it("does not scan same directory twice in the same scanning process", function () {
            var fileSystemScanner,
                fileSystemStrategy = new SynchronousFileSystemStrategy(),
                scanningProcess,
                scanningStrategy = new DirectoryScanningStrategy(/.*/);

            scanningStrategy.shouldScanDirectory = function (path, fs, cb) {
                cb(false);
            };

            fileSystemScanner = new FileSystemScanner(scanningStrategy, fileSystemStrategy);
            scanningProcess = fileSystemScanner.scanDirectory(__dirname);

            assert.ok(scanningProcess.isStopped());
            assert.deepEqual([__dirname], scanningProcess.foundDirectories);
            assert.deepEqual([__dirname], scanningProcess.foundPaths);

            scanningProcess = fileSystemScanner.scanDirectory(__dirname, scanningProcess);

            assert.ok(scanningProcess.isStopped());
            assert.deepEqual([__dirname], scanningProcess.foundDirectories);
            assert.deepEqual([__dirname], scanningProcess.foundPaths);
        });

        it("does not find directory", function () {
            var fileSystemScanner,
                fileSystemStrategy = new SynchronousFileSystemStrategy(),
                scanningProcess,
                scanningStrategy = new FileScanningStrategy(/.*/);

            fileSystemScanner = new FileSystemScanner(scanningStrategy, fileSystemStrategy);
            scanningProcess = fileSystemScanner.scanDirectory(global.MODULE_ROOT_TESTS + "/fixtures/search_site_1/directory_3");

            assert.ok(scanningProcess.isStopped());

            assert.deepEqual([], scanningProcess.foundFiles);
            assert.deepEqual([], scanningProcess.foundDirectories);
            assert.deepEqual([], scanningProcess.foundPaths);
        });

        it("finds exactly one directory", function () {
            var fileSystemScanner,
                fileSystemStrategy = new SynchronousFileSystemStrategy(),
                scanningProcess,
                scanningStrategy = new DirectoryScanningStrategy(/.*/);

            fileSystemScanner = new FileSystemScanner(scanningStrategy, fileSystemStrategy);

            scanningProcess = fileSystemScanner.scanDirectory(__dirname, {
                limit: 1
            });
            assert.ok(scanningProcess.isStopped());

            assert.deepEqual([], scanningProcess.foundFiles);
            assert.deepEqual([__dirname], scanningProcess.foundDirectories);
            assert.deepEqual([__dirname], scanningProcess.foundPaths);
        });

        it("tries to scan non existent directory", function () {
            var fileSystemScanner,
                fileSystemStrategy = new SynchronousFileSystemStrategy(),
                scanningProcess,
                scanningStrategy = new FileScanningStrategy(/.*/);

            fileSystemScanner = new FileSystemScanner(scanningStrategy, fileSystemStrategy);
            scanningProcess = fileSystemScanner.scanDirectory(global.MODULE_ROOT_TESTS + "/fixtures/non_existent_directory");

            assert.ok(scanningProcess.isStopped());

            assert.deepEqual([], scanningProcess.foundFiles);
            assert.deepEqual([], scanningProcess.foundDirectories);
            assert.deepEqual([], scanningProcess.foundPaths);
        });

    });

    describe("#scanFile", function () {

        it("does not scan same file twice in the same scanning process", function () {
            var fileSystemScanner,
                fileSystemStrategy = new SynchronousFileSystemStrategy(),
                scanningProcess,
                scanningStrategy = new FileScanningStrategy(/.*/);

            fileSystemScanner = new FileSystemScanner(scanningStrategy, fileSystemStrategy);
            scanningProcess = fileSystemScanner.scanFile(__filename);

            assert.ok(scanningProcess.isStopped());
            assert.deepEqual([__filename], scanningProcess.visitedFiles);
            assert.deepEqual([__filename], scanningProcess.visitedPaths);

            scanningProcess = fileSystemScanner.scanFile(__filename, scanningProcess);

            assert.ok(scanningProcess.isStopped());
            assert.deepEqual([__filename], scanningProcess.visitedFiles);
            assert.deepEqual([__filename], scanningProcess.visitedPaths);
        });

        it("does not find file", function () {
            var fileSystemScanner,
                fileSystemStrategy = new SynchronousFileSystemStrategy(),
                scanningProcess,
                scanningStrategy = new DirectoryScanningStrategy("required pattern");

            fileSystemScanner = new FileSystemScanner(scanningStrategy, fileSystemStrategy);
            scanningProcess = fileSystemScanner.scanFile(__filename);

            assert.ok(scanningProcess.isStopped());

            assert.deepEqual([], scanningProcess.foundFiles);
            assert.deepEqual([], scanningProcess.foundDirectories);
            assert.deepEqual([], scanningProcess.foundPaths);
        });

        it("finds file", function () {
            var fileSystemScanner,
                fileSystemStrategy = new SynchronousFileSystemStrategy(),
                scanningProcess,
                scanningStrategy = new FileScanningStrategy(/.*/);

            fileSystemScanner = new FileSystemScanner(scanningStrategy, fileSystemStrategy);
            scanningProcess = fileSystemScanner.scanFile(__filename);

            assert.ok(scanningProcess.isStopped());

            assert.deepEqual([__filename], scanningProcess.foundFiles);
            assert.deepEqual([], scanningProcess.foundDirectories);
            assert.deepEqual([__filename], scanningProcess.foundPaths);
        });

    });

    describe("#scanPath", function () {

        it("does not scan same path twice in the same scanning process", function () {
            var fileSystemScanner,
                fileSystemStrategy = new SynchronousFileSystemStrategy(),
                scanningProcess,
                scanningStrategy = new FileScanningStrategy(/.*/);

            fileSystemScanner = new FileSystemScanner(scanningStrategy, fileSystemStrategy);
            scanningProcess = fileSystemScanner.scanPath(__filename);

            assert.ok(scanningProcess.isStopped());
            assert.deepEqual([__filename], scanningProcess.visitedFiles);
            assert.deepEqual([__filename], scanningProcess.visitedPaths);

            scanningProcess = fileSystemScanner.scanPath(__filename, scanningProcess);

            assert.ok(scanningProcess.isStopped());
            assert.deepEqual([__filename], scanningProcess.visitedFiles);
            assert.deepEqual([__filename], scanningProcess.visitedPaths);
        });

        it("identifies one path as an exactly one directory", function () {
            var fileSystemScanner,
                fileSystemStrategy = new SynchronousFileSystemStrategy(),
                scanningProcess,
                scanningStrategy = new DirectoryScanningStrategy(/.*/);

            fileSystemScanner = new FileSystemScanner(scanningStrategy, fileSystemStrategy);
            scanningProcess = fileSystemScanner.scanPath(__dirname, {
                limit: 1
            });

            assert.ok(scanningProcess.isStopped());

            assert.deepEqual([], scanningProcess.foundFiles);
            assert.deepEqual([__dirname], scanningProcess.foundDirectories);
            assert.deepEqual([__dirname], scanningProcess.foundPaths);
        });

        it("identifies path as a file", function () {
            var fileSystemScanner,
                fileSystemStrategy = new SynchronousFileSystemStrategy(),
                scanningProcess,
                scanningStrategy = new DirectoryScanningStrategy(/.*/);

            fileSystemScanner = new FileSystemScanner(scanningStrategy, fileSystemStrategy);
            scanningProcess = fileSystemScanner.scanPath(__filename);

            assert.ok(scanningProcess.isStopped());

            assert.deepEqual([], scanningProcess.foundFiles);
            assert.deepEqual([], scanningProcess.foundDirectories);
            assert.deepEqual([], scanningProcess.foundPaths);
        });

        it("identifies path as a file and scans it", function () {
            var fileSystemScanner,
                fileSystemStrategy = new SynchronousFileSystemStrategy(),
                scanningProcess,
                scanningStrategy = new FileScanningStrategy(/.*/);

            fileSystemScanner = new FileSystemScanner(scanningStrategy, fileSystemStrategy);
            scanningProcess = fileSystemScanner.scanPath(__filename);

            assert.ok(scanningProcess.isStopped());

            assert.deepEqual([__filename], scanningProcess.foundFiles);
            assert.deepEqual([], scanningProcess.foundDirectories);
            assert.deepEqual([__filename], scanningProcess.foundPaths);
        });

        it("looks for specific directory", function () {
            var fileSystemScanner,
                fileSystemStrategy = new SynchronousFileSystemStrategy(),
                scanningProcess,
                scanningStrategy = new DirectoryScanningStrategy(/search_t(ar)?ge?t/);

            fileSystemScanner = new FileSystemScanner(scanningStrategy, fileSystemStrategy);
            scanningProcess = fileSystemScanner.scanPath(global.MODULE_ROOT_TESTS + p.sep + "fixtures");

            assert.ok(scanningProcess.isStopped());

            assert.deepEqual([], scanningProcess.foundFiles);
            assert.deepEqual(scanningProcess.foundPaths, scanningProcess.foundDirectories);

            assert.equal(3, scanningProcess.foundDirectories.length);
        });

        it("reports error on non-existent path", function () {
            var fileSystemScanner,
                fileSystemStrategy = new SynchronousFileSystemStrategy(),
                scanningStrategy = new FileScanningStrategy(/.*/);

            fileSystemScanner = new FileSystemScanner(scanningStrategy, fileSystemStrategy);
            fileSystemScanner.addListener(FileSystemScanner.EVENT_ERROR_CAUGHT, function (evt) {
                assert.ok(evt.err);
            });

            fileSystemScanner.scanPath(__filename + __filename);
        });

        it("reports async error on non-existent path", function (done) {
            var fileSystemScanner,
                fileSystemStrategy = new AsynchronousFileSystemStrategy(),
                scanningStrategy = new FileScanningStrategy(/.*/);

            fileSystemScanner = new FileSystemScanner(scanningStrategy, fileSystemStrategy);
            fileSystemScanner.addListener(FileSystemScanner.EVENT_ERROR_CAUGHT, function (evt) {
                assert.ok(evt.err);
                done();
            });

            fileSystemScanner.scanPath(__filename + __filename);
        });

        it("listens for file to be found", function (done) {
            var basename = "search_tgt_2.txt",
                fileSystemScanner,
                fileSystemStrategy = new AsynchronousFileSystemStrategy(),
                scanningStrategy = new FileScanningStrategy(basename);

            fileSystemScanner = new FileSystemScanner(scanningStrategy, fileSystemStrategy);
            fileSystemScanner.addListener(FileSystemScanner.EVENT_FOUND_FILE, function (evt) {
                assert.notEqual(evt.file, basename);
                assert.equal(p.basename(evt.file), basename);
                done();
            });

            fileSystemScanner.scanPath(global.MODULE_ROOT_TESTS + "/fixtures");
        });

        it("listens for all files to be found", function (done) {
            var fileSystemScanner,
                fileSystemStrategy = new AsynchronousFileSystemStrategy(),
                scanningProcess,
                scanningStrategy = new FileScanningStrategy(/.*/);

            fileSystemScanner = new FileSystemScanner(scanningStrategy, fileSystemStrategy);
            scanningProcess = fileSystemScanner.scanPath(global.MODULE_ROOT_TESTS + p.sep + "fixtures");
            scanningProcess.addListenerDone(function (evt) {
                assert.strictEqual(evt.foundFiles.length, 8);
                done();
            });
        });

        it("listens for specific number of files to be found", function (done) {
            var fileSystemScanner,
                fileSystemStrategy = new AsynchronousFileSystemStrategy(),
                scanningProcess,
                scanningStrategy = new FileScanningStrategy(/.*/);

            fileSystemScanner = new FileSystemScanner(scanningStrategy, fileSystemStrategy);
            scanningProcess = fileSystemScanner.scanPath(global.MODULE_ROOT_TESTS + p.sep + "fixtures", {
                limit: 3
            });
            scanningProcess.addListenerDone(function (evt) {
                assert.strictEqual(evt.foundFiles.length, 3);
                done();
            });
        });

    });

    describe("#scanPaths", function () {

        it("does not accept anything but array", function () {
            var fileSystemScanner,
                fileSystemStrategy = new SynchronousFileSystemStrategy(),
                scanningStrategy = new DirectoryScanningStrategy(/.*/);

            fileSystemScanner = new FileSystemScanner(scanningStrategy, fileSystemStrategy);

            assert.throws(function () {
                fileSystemScanner.scanPaths(__dirname);
            });
        });

        it("dispatches limited path list", function () {
            var fileSystemScanner,
                fileSystemStrategy = new SynchronousFileSystemStrategy(),
                scanningProcess,
                scanningStrategy = new DirectoryScanningStrategy(/.*/);

            fileSystemScanner = new FileSystemScanner(scanningStrategy, fileSystemStrategy);
            scanningProcess = fileSystemScanner.scanPaths([__dirname], {
                limit: 1
            });

            assert.ok(scanningProcess.isStopped());

            assert.deepEqual([], scanningProcess.foundFiles);
            assert.deepEqual([__dirname], scanningProcess.foundDirectories);
            assert.deepEqual([__dirname], scanningProcess.foundPaths);
        });

        it("handles parent paths correctly", function () {
            var fileSystemScanner,
                fileSystemStrategy = new SynchronousFileSystemStrategy(),
                paths = [],
                scanningProcess,
                scanningStrategy = new DirectoryScanningStrategy(/.*/);

            paths.push([global.MODULE_ROOT_TESTS, "fixtures", "search_site_1"].join(p.sep));
            paths.push([global.MODULE_ROOT_TESTS, "fixtures", "search_site_1", "directory_1"].join(p.sep));

            fileSystemScanner = new FileSystemScanner(scanningStrategy, fileSystemStrategy);
            scanningProcess = fileSystemScanner.scanPaths(paths, {
                limit: 2
            });

            assert.ok(scanningProcess.isStopped());

            assert.deepEqual([], scanningProcess.foundFiles);
            assert.deepEqual(paths, scanningProcess.foundDirectories);
            assert.deepEqual(paths, scanningProcess.foundPaths);
        });

        it("handles child path with done parent path", function () {
            var fileSystemScanner,
                fileSystemStrategy = new SynchronousFileSystemStrategy(),
                paths,
                scanningProcess,
                scanningStrategy = new DirectoryScanningStrategy(/.*/);

            // paths order matters here
            paths = [
                global.MODULE_ROOT_TESTS + "/fixtures/search_site_1",
                global.MODULE_ROOT_TESTS + "/fixtures/search_site_1/directory_2/directory_5"
            ];

            scanningStrategy.shouldScanDirectory = function (path, fs, cb) {
                cb(false);
            };

            fileSystemScanner = new FileSystemScanner(scanningStrategy, fileSystemStrategy);
            scanningProcess = fileSystemScanner.scanPaths(paths);

            assert.ok(scanningProcess.isStopped());

            assert.deepEqual([], scanningProcess.foundFiles);
            assert.deepEqual([], scanningProcess.foundDirectories);
            assert.deepEqual([], scanningProcess.foundPaths);
        });

    });

});
