/**
 * @license Copyright (c) 2013, CKSource - Frederico Knabben
 * For licensing, see LICENSE
 */

"use strict";

var Pattern, // constructor, function
    ScanningStrategy = require(__dirname + "/../ScanningStrategy"), // constructor, function
    p = require("path"),
    testPattern; // private, function

/**
 * @access private
 * @param {JsLoader/FileSystem/ScanningStrategy/Pattern} patternScanningStrategy
 * @param {string} input
 * @return {bool}
 * @this JsLoader/FileSystem/ScanningStrategy/Pattern
 */
testPattern = function (patternScanningStrategy, input) {
    var pattern = patternScanningStrategy.pattern;

    if (pattern instanceof RegExp) {
        return pattern.test(input);
    } else {
        return pattern === input;
    }
};

/**
 * @abstract
 * @auguments JsLoader/FileSystem/ScanningStrategy
 * @constructor
 * @param {RegExp|String} pattern
 */
Pattern = function (pattern) {
    ScanningStrategy.call(this);

    Object.defineProperty(this, "pattern", {

        /**
         * @return {mixed} pattern
         */
        get: function () {
            return pattern;
        },

        /**
         * @param {mixed} item
         * @return {void}
         * @throws {Error} if directory name pattern is not supported
         */
        set: function (item) {
            if (!this.isNamePatternSupported(item)) {
                throw new Error("Given directory name pattern is not supported.");
            }

            pattern = item;
        }

    });

    this.pattern = pattern;
};
Pattern.prototype = Object.create(ScanningStrategy.prototype);

/**
 * @param {string} path
 * @param {JsLoader/FileSystem/FileSystemStrategy} fileSystemStrategy
 * @param {JsLoader/FileSystem/ScanningStrategy~checkDirectoryCallback} callback
 * @return {void}
 */
Pattern.prototype.checkDirectory = function (path, fileSystemStrategy, callback) {
    var basename,
        resolvedPath;

    resolvedPath = p.resolve(path);
    basename = p.basename(resolvedPath);

    callback(testPattern(this, basename));
};

/**
 * @param {string} path
 * @param {JsLoader/FileSystem/FileSystemStrategy} fileSystemStrategy
 * @param {JsLoader/FileSystem/ScanningStrategy~checkFileCallback} callback
 * @return {void}
 */
Pattern.prototype.checkFile = function (path, fileSystemStrategy, callback) {
    var basename,
        resolvedPath;

    resolvedPath = p.resolve(path);
    basename = p.basename(resolvedPath);

    callback(testPattern(this, basename));
};

/**
 * @abstract
 * @param {mixed} pattern
 * @return {bool}
 */
Pattern.prototype.isNamePatternSupported = function (pattern) {
    return ("string" === typeof pattern) || (pattern instanceof RegExp);
};

module.exports = Pattern;
