/**
 * @license Copyright (c) 2013, CKSource - Frederico Knabben
 * For licensing, see LICENSE
 */

"use strict";

var EventDispatcher = require("event-dispatcher").EventDispatcher,
    ScanningConfiguration = require(__dirname + "/ScanningConfiguration"),
    ScanningBranch = require(__dirname + "/ScanningBranch"),
    ScanningBranchAggregator = require(__dirname + "/ScanningBranchAggregator"),
    ScanningProcess, // constructor, function
    config = require("configuration-processor").configurationProcessor,
    getOrSpawnBranch, // private, function
    onPathStop, // private, function
    onPathVisited, // private, function
    spawnBranch; // private, function

/**
 * @access private
 * @param {JsLoader/FileSystem/ScanningProcess} scanningProcess
 * @param {string} path
 * @return {JsLoader/FileSystem/ScanningBranch}
 */
getOrSpawnBranch = function (scanningProcess, path) {
    path = config.assertString(path);

    if (scanningProcess.hasBranch(path)) {
        return scanningProcess.getBranch(path);
    }

    return spawnBranch(scanningProcess);
};

/**
 * @access private
 * @param {JsLoader/FileSystem/ScanningProcess} scanningProcess
 * @param {string} path
 * @param {JsLoader/FileSystem/ScanningBranch} branch
 * @return {void}
 */
onPathStop = function (scanningProcess, path, branch) {
    if (scanningProcess.isStopped() && !scanningProcess.isExpectingAnyPath()) {
        scanningProcess.notifyStop();
    }
};

/**
 * @access private
 * @param {JsLoader/FileSystem/ScanningProcess} scanningProcess
 * @param {string} path
 * @param {JsLoader/FileSystem/ScanningBranch} branch
 * @return {void}
 */
onPathVisited = function (scanningProcess, path, branch) {
    if (scanningProcess.isExpectingPath(path)) {
        scanningProcess.unexpectPath(path);
    }
};

/**
 * @access private
 * @param {JsLoader/FileSystem/ScanningProcess} scanningProcess
 * @param {string} path
 * @return {JsLoader/FileSystem/ScanningBranch}
 */
spawnBranch = function (scanningProcess) {
    var branch = new ScanningBranch(scanningProcess.options);

    scanningProcess.branches.add(branch);
    branch.start();

    return branch;
};

/**
 * @auguments event-dispatcher/EventDispatcher
 * @constructor
 * @param {object} [options]
 */
ScanningProcess = function (options) {
    EventDispatcher.call(this);

    this.branches = new ScanningBranchAggregator(options);
    this.expectedPaths = {};
    this.options = new ScanningConfiguration(options);
};
ScanningProcess.prototype = Object.create(EventDispatcher.prototype);

/**
 * @event JsLoader/FileSystem/ScanningProcess#EVENT_STOP
 * @type {object}
 * @property {JsLoader/FileSystem/ScanningProcess} dispatcher
 * @property {array} foundDirectories
 * @property {array} foundFiles
 * @property {array} foundPaths
 * @property {array} visitedDirectories
 * @property {array} visitedFiles
 * @property {array} visitedPaths
 */
Object.defineProperty(ScanningProcess, "EVENT_STOP", {
    enumerable: true,
    value: "scanning_process.event.done"
});

Object.defineProperties(ScanningProcess.prototype, {
    foundDirectories: {
        enumerable: true,

        /**
         * @return {array}
         */
        get: function () {
            return this.branches.foundDirectories;
        }
    },
    foundFiles: {
        enumerable: true,

        /**
         * @return {array}
         */
        get: function () {
            return this.branches.foundFiles;
        }
    },
    foundPaths: {
        enumerable: true,

        /**
         * @return {array}
         */
        get: function () {
            return this.branches.foundPaths;
        }
    },
    visitedDirectories: {
        enumerable: true,

        /**
         * @return {array}
         */
        get: function () {
            return this.branches.visitedDirectories;
        }
    },
    visitedFiles: {
        enumerable: true,

        /**
         * @return {array}
         */
        get: function () {
            return this.branches.visitedFiles;
        }
    },
    visitedPaths: {
        enumerable: true,

        /**
         * @return {array}
         */
        get: function () {
            return this.branches.visitedPaths;
        }
    }
});

/**
 * @param {string} path
 * @return {JsLoader/FileSystem/ScanningBranch}
 */
ScanningProcess.prototype.getBranch = function (path) {
    var branch,
        branches = this.branches.list,
        deadParent,
        i;

    for (i = 0; i < branches.length; i += 1) {
        branch = branches[i];
        if (branch.coversPath(path)) {
            if (branch.isStopped()) {
                deadParent = true;
            } else {
                return branch;
            }
        }
    }

    if (deadParent && this.expectedPaths[path]) {
        // this situation may occur when scanning multiple paths or
        // scanning same branch multiple times using same scanning process
        // and current path's parent branch is stopped
        return spawnBranch(this);
    }

    throw new Error("There is no active branch that covers path: " + path);
};

/**
 * @param {function} listener
 * @return {void}
 */
ScanningProcess.prototype.addListenerDone = function (listener) {
    this.addListener(ScanningProcess.EVENT_STOP, listener);
};

/**
 * @param {string} path
 * @return {void}
 * @throws {Error} if path is already expected
 */
ScanningProcess.prototype.expectPath = function (path) {
    if (this.isExpectingPath(path)) {
        throw new Error("Path is already expected: " + path);
    }

    this.expectedPaths[path] = true;
};

/**
 * @return {array}
 */
ScanningProcess.prototype.getSupportedEvents = function () {
    return [
        ScanningProcess.EVENT_STOP
    ];
};

/**
 * @param {string} path
 * @return {bool}
 */
ScanningProcess.prototype.hasBranch = function (path) {
    path = config.assertString(path);

    return this.branches.list.some(function (branch) {
        return branch.coversPath(path);
    });
};

/**
 * @return {bool}
 */
ScanningProcess.prototype.isExpectingAnyPath = function () {
    var property;

    for (property in this.expectedPaths) {
        if (this.expectedPaths.hasOwnProperty(property)) {
            return true;
        }
    }

    return false;
};

/**
 * @param {string} path
 * @return {bool}
 */
ScanningProcess.prototype.isExpectingPath = function (path) {
    path = config.assertString(path);

    return !!this.expectedPaths[path];
};

/**
 * @return {bool}
 */
ScanningProcess.prototype.isStopped = function () {
    return !this.isExpectingAnyPath() && this.branches.isEveryStopped();
};

/**
 * @param {string} path
 * @return {bool}
 */
ScanningProcess.prototype.isDirectoryVisited = function (path) {
    return -1 !== this.visitedDirectories.indexOf(path);
};

/**
 * @param {string} path
 * @return {bool}
 */
ScanningProcess.prototype.isFileVisited = function (path) {
    return -1 !== this.visitedFiles.indexOf(path);
};

/**
 * @param {string} path
 * @return {bool}
 */
ScanningProcess.prototype.isPathVisited = function (path) {
    return -1 !== this.visitedPaths.indexOf(path);
};

/**
 * @return {void}
 */
ScanningProcess.prototype.notifyStop = function () {
    this.dispatch(ScanningProcess.EVENT_STOP, {
        foundDirectories: this.foundDirectories,
        foundFiles: this.foundFiles,
        foundPaths: this.foundPaths,
        visitedDirectories: this.visitedDirectories,
        visitedFiles: this.visitedFiles,
        visitedPaths: this.visitedPaths
    });
};

/**
 * @param {string} path
 * @return {bool}
 */
ScanningProcess.prototype.onDirectoryFound = function (path) {
    var branch = getOrSpawnBranch(this, path);

    branch.foundDirectories.push(path);
};

/**
 * @param {string} path
 * @return {bool}
 */
ScanningProcess.prototype.onDirectoryVisited = function (path) {
    var branch = getOrSpawnBranch(this, path);

    branch.visitedDirectories.push(path);

    onPathVisited(this, path, branch);
};

/**
 * @param {string} path
 * @return {bool}
 */
ScanningProcess.prototype.onFileFound = function (path) {
    var branch = getOrSpawnBranch(this, path);

    branch.foundFiles.push(path);
};

/**
 * @param {string} path
 * @return {bool}
 */
ScanningProcess.prototype.onFileVisited = function (path) {
    var branch = getOrSpawnBranch(this, path);

    branch.visitedFiles.push(path);

    onPathVisited(this, path, branch);
};

/**
 * @param {string} path
 * @return {bool}
 */
ScanningProcess.prototype.onPathDone = function (path) {
    var branch = this.getBranch(path);

    branch.done();

    onPathStop(this, path, branch);
};

/**
 * @param {string} path
 * @return {bool}
 */
ScanningProcess.prototype.onPathFailed = function (path) {
    var branch = this.getBranch(path);

    branch.fail();

    onPathStop(this, path, branch);
};

/**
 * @return {bool}
 */
ScanningProcess.prototype.shouldStop = function () {
    if (this.isStopped() || 0 === this.options.limit) {
        return false;
    }

    return this.foundPaths.length >= this.options.limit;
};

/**
 * @param {string} path
 * @return {bool}
 * @throws {Error} if path is not expected to be visited
 */
ScanningProcess.prototype.unexpectPath = function (path) {
    if (!this.isExpectingPath(path)) {
        throw new Error("Path is not expected to be visited: " + path);
    }

    delete this.expectedPaths[path];
};

module.exports = ScanningProcess;
