/**
 * @license Copyright (c) 2013, CKSource - Frederico Knabben
 * For licensing, see LICENSE
 */

"use strict";

var FileSystemStrategy = require(__dirname + "/../FileSystemStrategy"),
    Synchronous, // constructor, function
    fs = require("fs");

/**
 * @auguments JsLoader/FileSystem/FileSystemStrategy
 * @constructor
 */
Synchronous = function () {
    FileSystemStrategy.call(this);
};
Synchronous.prototype = Object.create(FileSystemStrategy.prototype);

/**
 * @param {string} path
 * @param {JsLoader/FileSystem/FileSystemStrategy~existsCallback} callback
 * @return {void}
 */
Synchronous.prototype.exists = function (path, callback) {
    callback(fs.existsSync(path));
};

/**
 * @param {string} path
 * @param {JsLoader/FileSystem/FileSystemStrategy~readDirCallback} callback
 * @return {void}
 */
Synchronous.prototype.readDir = function (path, callback) {
    var err,
        files;

    try {
        files = fs.readdirSync(path);
    } catch (e) {
        err = e;
    }

    callback(files, err);
};

/**
 * @param {string} path
 * @param {object} options
 * @param {JsLoader/FileSystem/FileSystemStrategy~readFileCallback} callback
 * @return {void}
 */
Synchronous.prototype.readFile = function (path, options, callback) {
    var content,
        err;

    try {
        content = fs.readFileSync(path, options);
    } catch (e) {
        err = e;
    }

    callback(content, err);
};

/**
 * @param {string} path
 * @param {JsLoader/FileSystem/FileSystemStrategy~realpathCallback} callback
 * @return {void}
 * @throws {Error}
 */
Synchronous.prototype.realpath = function (path, callback) {
    var err,
        realpath;

    try {
        realpath = fs.realpathSync(path);
    } catch (e) {
        err = e;
    }

    callback(realpath, err);
};

/**
 * @param {string} path
 * @param {JsLoader/FileSystem/FileSystemStrategy~statCallback} callback
 * @return {void}
 */
Synchronous.prototype.stat = function (path, callback) {
    var err,
        stats;

    try {
        stats = fs.statSync(path);
    } catch (e) {
        err = e;
    }

    callback(stats, err);
};

module.exports = Synchronous;
