/**
 * @license Copyright (c) 2013, CKSource - Frederico Knabben
 * For licensing, see LICENSE
 */

"use strict";

var flowInspector = require("flow-inspector"),
    p = require("path"),
    FileSystemStrategy, // constructor, function
    Task = flowInspector.Task,
    TaskAggregator = flowInspector.TaskAggregator;

/**
 * @abstract
 * @constructor
 */
FileSystemStrategy = function () {};

/**
 * @callback JsLoader/FileSystem/FileSystemStrategy~existsCallback
 * @param {bool} exists
 * @return {void}
 */

/**
 * @abstract
 * @param {string} path
 * @param {JsLoader/FileSystem/FileSystemStrategy~existsCallback} callback
 * @return {void}
 * @throws {Error}
 */
FileSystemStrategy.prototype.exists = function (path, callback) {
    throw new Error("This method must be implemented in child object.");
};

/**
 * @callback JsLoader/FileSystem/FileSystemStrategy~hasDirDirCallback
 * @param {bool} has
 * @param {object|void} err
 * @return {void}
 */

/**
 * @param {string} directory
 * @param {string} innerDirectory relative to directory
 * @param {JsLoader/FileSystem/FileSystemStrategy~hasDirDirCallback} callback
 * @return {void}
 */
FileSystemStrategy.prototype.hasDirDir = function (directory, innerDirectory, callback) {
    var that = this;

    this.hasDirPath(directory, innerDirectory, function (exists, err) {
        if (exists) {
            that.isDirectory(p.resolve(directory, innerDirectory), function (isDirectory, err) {
                callback(isDirectory, err);
            });
        } else {
            callback(exists, err);
        }
    });
};

/**
 * @callback JsLoader/FileSystem/FileSystemStrategy~hasDirFileCallback
 * @param {bool} has
 * @param {object|void} err
 * @return {void}
 */

/**
 * @param {string} directory
 * @param {string} file
 * @param {JsLoader/FileSystem/FileSystemStrategy~hasDirFileCallback} callback
 * @return {void}
 */
FileSystemStrategy.prototype.hasDirFile = function (directory, file, callback) {
    var that = this;

    this.hasDirPath(directory, file, function (exists, err) {
        if (exists) {
            that.isFile(p.resolve(directory, file), function (isFile, err) {
                callback(isFile, err);
            });
        } else {
            callback(exists, err);
        }
    });
};

/**
 * @callback JsLoader/FileSystem/FileSystemStrategy~hasDirPathCallback
 * @param {bool} has
 * @param {object|void} err
 * @return {void}
 */

/**
 * @param {string} directory
 * @param {string} path
 * @param {JsLoader/FileSystem/FileSystemStrategy~hasDirPathCallback} callback
 * @return {void}
 * @throws {Error} if given path is not a directory
 */
FileSystemStrategy.prototype.hasDirPath = function (directory, path, callback) {
    var that = this;

    this.isDirectory(directory, function (isDirectory, err) {
        if (!err && !isDirectory) {
            err = new Error("Given path is not a directory: " + directory);
        }

        if (err) {
            return void callback(false, err);
        }

        that.exists(p.resolve(directory, path), function (exists) {
            callback(exists, err);
        });
    });
};

/**
 * @callback JsLoader/FileSystem/FileSystemStrategy~isDirectoryCallback
 * @param {bool} isDirectory
 * @param {object|void} err
 * @return {void}
 */

/**
 * @param {string} path
 * @param {JsLoader/FileSystem/FileSystemStrategy~isDirectoryCallback} callback
 * @return {void}
 */
FileSystemStrategy.prototype.isDirectory = function (path, callback) {
    this.stat(path, function (stat, err) {
        var isDirectory;

        if (err) {
            isDirectory = false;
        } else {
            isDirectory = stat.isDirectory();
        }

        callback(isDirectory, err);
    });
};

/**
 * @callback JsLoader/FileSystem/FileSystemStrategy~isFileCallback
 * @param {bool} isFile
 * @param {object|void} err
 * @return {void}
 */

/**
 * @param {string} path
 * @param {JsLoader/FileSystem/FileSystemStrategy~isFileCallback} callback
 * @return {void}
 */
FileSystemStrategy.prototype.isFile = function (path, callback) {
    this.stat(path, function (stat, err) {
        var isFile;

        if (err) {
            isFile = false;
        } else {
            isFile = stat.isFile();
        }

        callback(isFile, err);
    });
};

/**
 * @callback JsLoader/FileSystem/FileSystemStrategy~readDirCallback
 * @param {array} files
 * @param {object|void} err
 * @return {void}
 */

/**
 * @abstract
 * @param {string} path
 * @param {JsLoader/FileSystem/FileSystemStrategy~readDirCallback} callback
 * @return {void}
 * @throws {Error}
 */
FileSystemStrategy.prototype.readDir = function (path, callback) {
    throw new Error("This method must be implemented in child object.");
};

/**
 * @callback JsLoader/FileSystem/FileSystemStrategy~readDirsCallback
 * @param {array} files
 * @param {object|void} err
 * @return {void}
 */

/**
 * @param {array} paths
 * @param {JsLoader/FileSystem/FileSystemStrategy~readDirsCallback} callback
 * @return {void}
 * @throws {Error}
 */
FileSystemStrategy.prototype.readDirs = function (paths, callback) {
    var i,
        readTask,
        readTaskAggregator = new TaskAggregator(),
        ret = {},
        onDirectoryRead;

    onDirectoryRead = function (readTask, path, files, err) {
        if (err) {
            return void readTask.fail();
        }

        ret[path] = files;
        readTask.done();
    };

    for (i = 0; i < paths.length; i += 1) {
        readTask = new Task();
        readTask.start();

        readTaskAggregator.add(readTask);
        this.readDir(paths[i], onDirectoryRead.bind(null, readTask, paths[i]));
    }

    if (readTaskAggregator.isEveryStopped()) {
        callback(ret);
    } else {
        readTaskAggregator.addListenerEveryStop(function (evt) {
            callback(ret);
        });
    }
};

/**
 * @callback JsLoader/FileSystem/FileSystemStrategy~readFileCallback
 * @param {string} content
 * @param {object|void} err
 * @return {void}
 */

/**
 * @abstract
 * @param {string} path
 * @param {JsLoader/FileSystem/FileSystemStrategy~readFileCallback} callback
 * @return {void}
 * @throws {Error}
 */
FileSystemStrategy.prototype.readFile = function (path, callback) {
    throw new Error("This method must be implemented in child object.");
};

/**
 * @callback JsLoader/FileSystem/FileSystemStrategy~realpathCallback
 * @param {string} realpath
 * @param {object|void} err
 * @return {void}
 */

/**
 * @abstract
 * @param {string} path
 * @param {JsLoader/FileSystem/FileSystemStrategy~realpathCallback} callback
 * @return {void}
 * @throws {Error}
 */
FileSystemStrategy.prototype.realpath = function (path, callback) {
    throw new Error("This method must be implemented in child object.");
};

/**
 * @callback JsLoader/FileSystem/FileSystemStrategy~statCallback
 * @param {fs.Stats} stat or any object compatible with fs.Stats by duck typing
 * @param {object|void} err
 * @return {void}
 */

/**
 * @abstract
 * @param {string} path
 * @param {JsLoader/FileSystem/FileSystemStrategy~statCallback} callback
 * @return {void}
 * @throws {Error}
 */
FileSystemStrategy.prototype.stat = function (path, callback) {
    throw new Error("This method must be implemented in child object.");
};

/**
 * @callback JsLoader/FileSystem/FileSystemStrategy~statPathsCallback
 * @param {array} files
 * @param {object|void} err
 * @return {void}
 */

/**
 * @param {array} paths
 * @param {JsLoader/FileSystem/FileSystemStrategy~statPathsCallback} callback
 * @return {void}
 * @throws {Error}
 */
FileSystemStrategy.prototype.statPaths = function (paths, callback) {
    var i,
        statTask,
        statTaskAggregator = new TaskAggregator(),
        ret = {},
        onPathStat;

    onPathStat = function (statTask, path, stat, err) {
        if (err) {
            return void statTask.fail();
        }

        ret[path] = stat;
        statTask.done();
    };

    for (i = 0; i < paths.length; i += 1) {
        statTask = new Task();
        statTask.start();

        statTaskAggregator.add(statTask);
        this.stat(paths[i], onPathStat.bind(null, statTask, paths[i]));
    }

    if (statTaskAggregator.isEveryStopped()) {
        callback(ret);
    } else {
        statTaskAggregator.addListenerEveryStop(function (evt) {
            callback(ret);
        });
    }
};

module.exports = FileSystemStrategy;
