/**
 * @license Copyright (c) 2013, CKSource - Frederico Knabben
 * For licensing, see LICENSE
 */

"use strict";

var FileSystemScanner = require(__dirname + "/../FileSystemScanner"),
    Remembering, // constructor, function
    config = require("configuration-processor").configurationProcessor;

/**
 * @auguments JsLoader/FileSystem/FileSystemScanner
 * @constructor
 * @param {JsLoader/FileSystem/ScanningStrategy} scanningStrategy
 * @param {JsLoader/FileSystem/FileSystemStrategy} fileSystemStrategy
 */
Remembering = function (scanningStrategy, fileSystemStrategy) {
    FileSystemScanner.call(this, scanningStrategy, fileSystemStrategy);

    this.scannedDirectories = {};
    this.scannedFiles = {};
    this.scannedPaths = {};
};
Remembering.prototype = Object.create(FileSystemScanner.prototype);

/**
 * @param {string} directory
 * @return {bool}
 */
Remembering.prototype.isDirectoryScanned = function (directory) {
    return !!this.scannedDirectories[directory];
};

/**
 * @param {string} file
 * @return {bool}
 */
Remembering.prototype.isFileScanned = function (file) {
    return !!this.scannedFiles[file];
};

/**
 * @param {string} path
 * @return {bool}
 */
Remembering.prototype.isPathScanned = function (path) {
    return !!this.scannedPaths[path];
};

/**
 * @param {string} path
 * @param {JsLoader/FileSystem/ScanningProcess} [scanningProcess]
 * @return {JsLoader/FileSystem/ScanningProcess}
 */
Remembering.prototype.scanDirectory = function (path, scanningProcess) {
    if (this.scannedDirectories[path]) {
        return this.scannedDirectories[path];
    }

    scanningProcess = FileSystemScanner.prototype.scanDirectory.call(this, path, scanningProcess);

    this.scannedDirectories[path] = scanningProcess;
    this.scannedPaths[path] = scanningProcess;

    return scanningProcess;
};

/**
 * @param {string} path
 * @param {JsLoader/FileSystem/ScanningProcess} [scanningProcess]
 * @return {JsLoader/FileSystem/ScanningProcess}
 */
Remembering.prototype.scanFile = function (path, scanningProcess) {
    if (this.scannedFiles[path]) {
        return this.scannedFiles[path];
    }

    scanningProcess = FileSystemScanner.prototype.scanFile.call(this, path, scanningProcess);

    this.scannedFiles[path] = scanningProcess;
    this.scannedPaths[path] = scanningProcess;

    return scanningProcess;
};

/**
 * @param {string} path
 * @param {JsLoader/FileSystem/ScanningProcess} [scanningProcess]
 * @return {JsLoader/FileSystem/ScanningProcess}
 */
Remembering.prototype.scanPath = function (path, scanningProcess) {
    if (this.scannedPaths[path]) {
        return this.scannedPaths[path];
    }

    scanningProcess = FileSystemScanner.prototype.scanPath.call(this, path, scanningProcess);
    this.scannedPaths[path] = scanningProcess;

    return scanningProcess;
};

/**
 * @param {array} paths
 * @param {JsLoader/FileSystem/ScanningProcess} [scanningProcess]
 * @return {JsLoader/FileSystem/ScanningProcess}
 * @throws {Error} if paths list is not an array
 */
Remembering.prototype.scanPaths = function (paths, scanningProcess) {
    paths = config.assertArrayEach(config.expectString(), paths);
    paths = paths.filter(function (path, index, paths) {
        return !this.scannedPaths[path];
    }, this);

    return FileSystemScanner.prototype.scanPaths.call(this, paths, scanningProcess);
};

module.exports = Remembering;
