/**
 * @license Copyright (c) 2013, CKSource - Frederico Knabben
 * For licensing, see LICENSE
 */

"use strict";

var EventDispatcher = require("event-dispatcher").EventDispatcher,
    FileSystemScanner, // constructor, function
    FileSystemStrategy = require(__dirname + "/FileSystemStrategy"),
    ScanningProcess = require(__dirname + "/ScanningProcess"),
    ScanningStrategy = require(__dirname + "/ScanningStrategy"),

    config = require("configuration-processor").configurationProcessor,
    p = require("path"),

    doScanDirectory, // private, function
    handleError, // private, function
    onCheckDirectory, // private, function
    onCheckFile, // private, function
    onShouldScanDirectory, // private, function
    onShouldScanFile, // private, function
    onShouldScanPath, // private, function
    sanitizeScanningProcess; // private, function

/**
 * @access private
 * @param {JsLoader/FileSystem/FileSystemScanner} fileSystemScanner
 * @param {JsLoader/FileSystem/ScanningProcess} scanningProcess
 * @param {string} path
 * @param {bool} shouldScan
 * @return {void}
 */
doScanDirectory = function (fileSystemScanner, scanningProcess, path) {
    fileSystemScanner.fileSystemStrategy.readDir(path, function (files, err) {
        var i;

        if (err) {
            return void handleError(fileSystemScanner, path, scanningProcess, err);
        }

        if (files.length < 1) {
            // branch stops on empty directory or file
            return void scanningProcess.onPathDone(path);
        }

        for (i = 0; i < files.length; i += 1) {
            files[i] = p.resolve(path, files[i]);
        }

        fileSystemScanner.scanPaths(files, scanningProcess);
    });
};

/**
 * Some errors might not be catchable if asynchronous filesystem strategy is
 * used. They need to be handled in other way.
 *
 * @access private
 * @param {JsLoader/FileSystem/FileSystemScanner} fileSystemScanner
 * @param {string} path path that caused the error
 * @param {JsLoader/FileSystem/ScanningProcess} scanningProcess
 * @param {object} err
 * @return {void}
 */
handleError = function (fileSystemScanner, path, scanningProcess, err) {
    if (scanningProcess.isExpectingPath(path)) {
        // this path is only expected to be visited and was not visited
        // because whole branch failed; most probably path failed while
        // executing "stat" and it was not possible to determine if this is a
        // file or directory or something else
        scanningProcess.unexpectPath(path);
    } else {
        // path was visited but something failed during internal processing
        scanningProcess.onPathFailed(path);
    }
    fileSystemScanner.notifyErrorCaught(err, path);
};

/**
 * @access private
 * @param {JsLoader/FileSystem/FileSystemScanner} fileSystemScanner
 * @param {JsLoader/FileSystem/ScanningProcess} scanningProcess
 * @param {JsLoader/FileSystem/FileSystemStrategy} fileSystemStrategy
 * @param {string} path
 * @param {bool} isFound
 * @return {void}
 */
onCheckDirectory = function (fileSystemScanner, scanningProcess, path, fileSystemStrategy, isFound) {
    if (scanningProcess.shouldStop()) {
        return void scanningProcess.onPathDone(path);
    }

    if (isFound) {
        scanningProcess.onDirectoryFound(path);
        fileSystemScanner.notifyDirectoryFound(path);
    }

    doScanDirectory(fileSystemScanner, scanningProcess, path);
};

/**
 * @access private
 * @param {JsLoader/FileSystem/FileSystemScanner} fileSystemScanner
 * @param {JsLoader/FileSystem/ScanningProcess} scanningProcess
 * @param {string} path
 * @param {JsLoader/FileSystem/FileSystemStrategy} fileSystemStrategy
 * @param {bool} isFound
 * @return {void}
 */
onCheckFile = function (fileSystemScanner, scanningProcess, path, fileSystemStrategy, isFound) {
    if (isFound && !scanningProcess.shouldStop()) {
        scanningProcess.onFileFound(path);
        fileSystemScanner.notifyFileFound(path);
    }

    scanningProcess.onPathDone(path);
};

/**
 * @access private
 * @param {JsLoader/FileSystem/FileSystemScanner} fileSystemScanner
 * @param {JsLoader/FileSystem/ScanningProcess} scanningProcess
 * @param {string} path
 * @param {JsLoader/FileSystem/FileSystemStrategy} fileSystemStrategy
 * @param {bool} shouldScan
 * @return {void}
 */
onShouldScanDirectory = function (fileSystemScanner, scanningProcess, path, fileSystemStrategy, shouldScan) {
    if (shouldScan) {
        fileSystemScanner.scanDirectory(path, scanningProcess);
    } else {
        scanningProcess.onDirectoryVisited(path);
        scanningProcess.onPathDone(path);
    }
};

/**
 * @access private
 * @param {JsLoader/FileSystem/FileSystemScanner} fileSystemScanner
 * @param {JsLoader/FileSystem/ScanningProcess} scanningProcess
 * @param {string} path
 * @param {JsLoader/FileSystem/FileSystemStrategy} fileSystemStrategy
 * @param {bool} shouldScan
 * @return {void}
 */
onShouldScanFile = function (fileSystemScanner, scanningProcess, path, fileSystemStrategy, shouldScan) {
    if (shouldScan) {
        fileSystemScanner.scanFile(path, scanningProcess);
    } else {
        scanningProcess.onFileVisited(path);
        scanningProcess.onPathDone(path);
    }
};

/**
 * @access private
 * @param {JsLoader/FileSystem/FileSystemScanner} fileSystemScanner
 * @param {JsLoader/FileSystem/ScanningProcess} scanningProcess
 * @param {string} path
 * @param {JsLoader/FileSystem/FileSystemStrategy} fileSystemStrategy
 * @param {bool} shouldScan
 * @return {void}
 */
onShouldScanPath = function (fileSystemScanner, scanningProcess, path, fileSystemStrategy, shouldScan) {
    if (shouldScan) {
        fileSystemScanner.scanPath(path, scanningProcess);
    } else {
        scanningProcess.unexpectPath(path);
    }
};

/**
 * @access private
 * @param {JsLoader/FileSystem/FileSystemScanner} fileSystemScanner
 * @param {JsLoader/FileSystem/ScanningProcess} scanningProcess
 * @return {JsLoader/FileSystem/ScanningProcess}
 */
sanitizeScanningProcess = function (fileSystemScanner, scanningProcess) {
    var options;

    if (!(scanningProcess instanceof ScanningProcess)) {
        options = (scanningProcess || {});

        scanningProcess = new ScanningProcess(options);
    }

    return scanningProcess;
};

/**
 * @auguments event-dispatcher/EventDispatcher
 * @constructor
 * @param {JsLoader/FileSystem/ScanningStrategy} scanningStrategy
 * @param {JsLoader/FileSystem/FileSystemStrategy} fileSystemStrategy
 */
FileSystemScanner = function (scanningStrategy, fileSystemStrategy) {
    EventDispatcher.call(this);

    Object.defineProperty(this, "fileSystemStrategy", {

        /**
         * @return {JsLoader/FileSystem/ScanningStrategy}
         */
        get: function () {
            return fileSystemStrategy;
        },

        /**
         * @param {JsLoader/FileSystem/FileSystemStrategy} item
         * @return {void}
         * @throws {Error} if argument is invalid
         */
        set: function (item) {
            if (!(item instanceof FileSystemStrategy)) {
                throw new Error("Expected JsLoader/FileSystem/FileSystemStrategy as an argument.");
            }

            fileSystemStrategy = item;
        }

    });

    Object.defineProperty(this, "scanningStrategy", {

        /**
         * @return {JsLoader/FileSystem/ScanningStrategy}
         */
        get: function () {
            return scanningStrategy;
        },

        /**
         * @param {JsLoader/FileSystem/ScanningStrategy} item
         * @return {void}
         * @throws {Error} if argument is invalid
         * @throws {Error} if given scanning strategy does not look for directory nor file
         */
        set: function (item) {
            if (!(item instanceof ScanningStrategy)) {
                throw new Error("Expected JsLoader/FileSystem/ScanningStrategy as an argument.");
            }

            if (!scanningStrategy.isLookingForDirectory && !scanningStrategy.isLookingForFile) {
                throw new Error("Given scanning strategy does not look for directory nor file.");
            }

            scanningStrategy = item;
        }

    });

    this.fileSystemStrategy = fileSystemStrategy;
    this.scanningStrategy = scanningStrategy;
};
FileSystemScanner.prototype = Object.create(EventDispatcher.prototype);

/**
 * @event JsLoader/FileSystem/FileSystemScanner#EVENT_ERROR_CAUGHT
 * @type {object}
 * @property {JsLoader/FileSystem/FileSystemScanner} dispatcher
 * @property {object} err
 */
Object.defineProperty(FileSystemScanner, "EVENT_ERROR_CAUGHT", {
    value: "filesystemscanner.event.error.caught"
});

/**
 * @event JsLoader/FileSystem/FileSystemScanner#EVENT_FOUND_DIRECTORY
 * @type {object}
 * @property {JsLoader/FileSystem/FileSystemScanner} dispatcher
 * @property {string} directory
 */
Object.defineProperty(FileSystemScanner, "EVENT_FOUND_DIRECTORY", {
    value: "filesystemscanner.event.found.directory"
});

/**
 * @event JsLoader/FileSystem/FileSystemScanner#EVENT_FOUND_FILE
 * @type {object}
 * @property {JsLoader/FileSystem/FileSystemScanner} dispatcher
 * @property {string} file
 */
Object.defineProperty(FileSystemScanner, "EVENT_FOUND_FILE", {
    value: "filesystemscanner.event.found.file"
});

/**
 * @event JsLoader/FileSystem/FileSystemScanner#EVENT_FOUND_PATH
 * @type {object}
 * @property {JsLoader/FileSystem/FileSystemScanner} dispatcher
 * @property {string} path
 */
Object.defineProperty(FileSystemScanner, "EVENT_FOUND_PATH", {
    value: "filesystemscanner.event.found.path"
});

/**
 * @return {array}
 */
FileSystemScanner.prototype.getSupportedEvents = function () {
    return [
        FileSystemScanner.EVENT_ERROR_CAUGHT,
        FileSystemScanner.EVENT_FOUND_DIRECTORY,
        FileSystemScanner.EVENT_FOUND_FILE,
        FileSystemScanner.EVENT_FOUND_PATH
    ];
};

/**
 * @fires JsLoader/FileSystem/FileSystemScanner#EVENT_FOUND_DIRECTORY
 * @param {string} path
 * @return {void}
 */
FileSystemScanner.prototype.notifyDirectoryFound = function (path) {
    this.dispatch(FileSystemScanner.EVENT_FOUND_DIRECTORY, {
        directory: path,
        dispatcher: this
    });
    this.notifyPathFound(path);
};

/**
 * @fires JsLoader/FileSystem/FileSystemScanner#EVENT_ERROR_CAUGHT
 * @param {object} err
 * @param {string|void} [path] path that caused the error
 * @return {void}
 */
FileSystemScanner.prototype.notifyErrorCaught = function (err, path) {
    this.dispatch(FileSystemScanner.EVENT_ERROR_CAUGHT, {
        dispatcher: this,
        err: err,
        path: path
    });
};

/**
 * @fires JsLoader/FileSystem/FileSystemScanner#EVENT_FOUND_FILE
 * @param {string} path
 * @return {void}
 */
FileSystemScanner.prototype.notifyFileFound = function (path) {
    this.dispatch(FileSystemScanner.EVENT_FOUND_FILE, {
        file: path,
        dispatcher: this
    });
    this.notifyPathFound(path);
};

/**
 * @fires JsLoader/FileSystem/FileSystemScanner#EVENT_FOUND_PATH
 * @param {string} path
 * @return {void}
 */
FileSystemScanner.prototype.notifyPathFound = function (path) {
    this.dispatch(FileSystemScanner.EVENT_FOUND_PATH, {
        dispatcher: this,
        path: path
    });
};

/**
 * @param {string} path
 * @param {JsLoader/FileSystem/ScanningProcess} [scanningProcess]
 * @return {JsLoader/FileSystem/ScanningProcess}
 */
FileSystemScanner.prototype.scanDirectory = function (path, scanningProcess) {
    var cb;

    scanningProcess = sanitizeScanningProcess(this, scanningProcess);
    if (scanningProcess.isDirectoryVisited(path)) {
        return scanningProcess;
    }

    scanningProcess.onDirectoryVisited(path);

    cb = onCheckDirectory.bind(null, this, scanningProcess, path, this.fileSystemStrategy);
    this.scanningStrategy.checkDirectory(path, this.fileSystemStrategy, cb);

    return scanningProcess;
};

/**
 * @param {string} path
 * @param {JsLoader/FileSystem/ScanningProcess} [scanningProcess]
 * @return {JsLoader/FileSystem/ScanningProcess}
 */
FileSystemScanner.prototype.scanFile = function (path, scanningProcess) {
    var cb;

    scanningProcess = sanitizeScanningProcess(this, scanningProcess);
    if (scanningProcess.isFileVisited(path)) {
        return scanningProcess;
    }

    scanningProcess.onFileVisited(path);

    if (!this.scanningStrategy.isLookingForFile) {
        scanningProcess.onPathDone(path);

        return scanningProcess;
    }

    cb = onCheckFile.bind(null, this, scanningProcess, path, this.fileSystemStrategy);
    this.scanningStrategy.checkFile(path, this.fileSystemStrategy, cb);

    return scanningProcess;
};

/**
 * @param {string} path
 * @param {JsLoader/FileSystem/ScanningProcess} [scanningProcess]
 * @return {JsLoader/FileSystem/ScanningProcess}
 */
FileSystemScanner.prototype.scanPath = function (path, scanningProcess) {
    var cb,
        fileSystemStrategy = this.fileSystemStrategy,
        scanningStrategy = this.scanningStrategy,
        that = this;

    scanningProcess = sanitizeScanningProcess(this, scanningProcess);
    if (scanningProcess.isPathVisited(path)) {
        return scanningProcess;
    }
    if (!scanningProcess.isExpectingPath(path)) {
        scanningProcess.expectPath(path);
    }

    fileSystemStrategy.stat(path, function (stat, err) {
        if (err) {
            return void handleError(that, path, scanningProcess, err);
        }

        if (stat.isDirectory()) {
            // even if scanning strategy does not look for directory,
            // directories should be scanned
            cb = onShouldScanDirectory.bind(null, that, scanningProcess, path, fileSystemStrategy);
            scanningStrategy.shouldScanDirectory(path, fileSystemStrategy, cb);
        } else if (stat.isFile()) {
            // file check must be done because it may also be FIFO, socket,
            // block device, character device, etc
            if (scanningStrategy.isLookingForFile) {
                cb = onShouldScanFile.bind(null, that, scanningProcess, path, fileSystemStrategy);
                scanningStrategy.shouldScanFile(path, fileSystemStrategy, cb);
            } else {
                // filesystem scanner is looking only for directories and this
                // path contains file at the end so this is a short visit
                scanningProcess.onFileVisited(path);
                scanningProcess.onPathDone(path);
            }
        }
    });

    return scanningProcess;
};

/**
 * @param {array} paths
 * @param {JsLoader/FileSystem/ScanningProcess} [scanningProcess]
 * @return {JsLoader/FileSystem/ScanningProcess}
 * @throws {Error} if paths list is not an array
 */
FileSystemScanner.prototype.scanPaths = function (paths, scanningProcess) {
    var cb,
        i;

    paths = config.assertArrayEach(config.expectString(), paths);
    scanningProcess = sanitizeScanningProcess(this, scanningProcess);

    for (i = 0; i < paths.length; i += 1) {
        if (!scanningProcess.isPathVisited(paths[i]) && !scanningProcess.isExpectingPath(paths[i])) {
            scanningProcess.expectPath(paths[i]);
            cb = onShouldScanPath.bind(null, this, scanningProcess, paths[i], this.fileSystemStrategy);
            this.scanningStrategy.shouldScanPath(paths[i], this.fileSystemStrategy, cb);
        }
    }

    return scanningProcess;
};

module.exports = FileSystemScanner;
