/**
 * @license Copyright (c) 2013, CKSource - Frederico Knabben
 * For licensing, see LICENSE
 */

"use strict";

var AsynchronousFileSystemStrategy = require(__dirname + "/../scripts/JsLoader/FileSystem/FileSystemStrategy/Asynchronous"),
    DirectoryScanningStrategy = require(__dirname + "/../scripts/JsLoader/FileSystem/ScanningStrategy/Pattern/Directory"),
    FileScanningStrategy = require(__dirname + "/../scripts/JsLoader/FileSystem/ScanningStrategy/Pattern/File"),
    FileSystemScanner = require(__dirname + "/../scripts/JsLoader/FileSystem/FileSystemScanner"),
    FileSystemStrategy = require(__dirname + "/../scripts/JsLoader/FileSystem/FileSystemStrategy"),
    PatternScanningStrategy = require(__dirname + "/../scripts/JsLoader/FileSystem/ScanningStrategy/Pattern"),
    RememberingFileSystemScanner = require(__dirname + "/../scripts/JsLoader/FileSystem/FileSystemScanner/Remembering"),
    ScanningConfiguration = require(__dirname + "/../scripts/JsLoader/FileSystem/ScanningConfiguration"),
    ScanningResult = require(__dirname + "/../scripts/JsLoader/FileSystem/ScanningResult"),
    ScanningStrategy = require(__dirname + "/../scripts/JsLoader/FileSystem/ScanningStrategy"),
    SynchronousFileSystemStrategy = require(__dirname + "/../scripts/JsLoader/FileSystem/FileSystemStrategy/Synchronous"),
    asynchronousFileSystemStrategy = new AsynchronousFileSystemStrategy(),
    config = require("configuration-processor").configurationProcessor,
    doScanPath, // private, function
    doScanPathSync, // private, function
    optimizeScanningStrategy, // private, function
    sanitizeCallback, // private, function
    sanitizeOptions, // private, function
    sanitizePath, // private, function
    scanPath, // function
    scanPathSync, // function
    synchronousFileSystemStrategy = new SynchronousFileSystemStrategy();

/**
 * @access private
 * @param {object} options
 * @return {JsLoader/FileSystem/ScanningStrategy}
 * @throws {Error}
 */
optimizeScanningStrategy = function (options) {
    if (!options.isLookingForDirectory && !options.isLookingForFile) {
        throw new Error("Configuration indicates that you are not looking for directory nor file.");
    }

    if (options.isLookingForDirectory && options.isLookingForFile) {
        return new PatternScanningStrategy(options.pattern);
    }

    if (options.isLookingForDirectory && !options.isLookingForFile) {
        return new DirectoryScanningStrategy(options.pattern);
    }

    return new FileScanningStrategy(options.pattern);
};

/**
 * @callback {scannerCallback}
 * @param {JsLoader/FileSystem/ScanningProcess#EVENT_DONE} evt
 */

/**
 * @access private
 * @param {string} path
 * @param {object} options
 * @param {scannerCallback} [callback]
 * @return {void}
 */
doScanPath = function (path, options, callback) {
    var fileSystemScanner = new FileSystemScanner(optimizeScanningStrategy(options), asynchronousFileSystemStrategy),
        scanningProcess;

    scanningProcess = fileSystemScanner.scanPath(path, options);

    if (callback) {
        scanningProcess.addListenerDone(function (evt) {
            callback(evt);
        });
    }

    return scanningProcess;
};

/**
 * @access private
 * @param {string} path
 * @param {object} options
 * @return {array}
 * @throws {Error}
 */
doScanPathSync = function (path, options) {
    var fileSystemScanner = new FileSystemScanner(optimizeScanningStrategy(options), synchronousFileSystemStrategy);

    return fileSystemScanner.scanPath(path);
};

/**
 * This function processess direct module user input.
 *
 * @access private
 * @param {mixed} callback
 * @return {funtion}
 */
sanitizeCallback = function (callback) {
    return config.assertFunction(callback);
};

/**
 * This function processess direct module user input.
 *
 * @access private
 * @param {mixed} options
 * @return {JsLoader/FileSystem/ScanningConfiguration}
 */
sanitizeOptions = function (options) {
    if (options instanceof RegExp || "string" === typeof options) {
        options = {
            pattern: options
        };
    } else if (!options) {
        options = {};
    }

    return new ScanningConfiguration(options);
};

/**
 * This function processess direct module user input.
 *
 * @access private
 * @param {mixed} path
 * @return {object}
 * @throws {Error} if given path is not a string
 */
sanitizePath = function (path) {
    return config.assertString(path);
};

/**
 * @access public
 * @param {string} path
 * @param {object} [options]
 * @param {scannerCallback} [callback]
 * @return {JsLoader/FileSystem/ScanningProcess}
 * @throws {Error} if invalid number of arguments given
 */
scanPath = function (path, arg2, arg3) {
    switch (arguments.length) {
    case 1:
        return doScanPath(sanitizePath(path), sanitizeOptions({}));
    case 2:
        return doScanPath(sanitizePath(path), sanitizeOptions({}), sanitizeCallback(arg2));
    case 3:
        return doScanPath(sanitizePath(path), sanitizeOptions(arg2), sanitizeCallback(arg3));
    }

    throw new Error("This function takes at least one argument.");
};

/**
 * @access public
 * @param {string} path
 * @param {object} [options]
 * @return {array}
 */
scanPathSync = function (path, options) {
    return doScanPathSync(sanitizePath(path), sanitizeOptions(options));
};

module.exports = {
    // classess
    AsynchronousFileSystemStrategy: AsynchronousFileSystemStrategy,
    DirectoryScanningStrategy: DirectoryScanningStrategy,
    FileScanningStrategy: FileScanningStrategy,
    FileSystemScanner: FileSystemScanner,
    FileSystemStrategy: FileSystemStrategy,
    PatternScanningStrategy: PatternScanningStrategy,
    RememberingFileSystemScanner: RememberingFileSystemScanner,
    ScanningResult: ScanningResult,
    ScanningStrategy: ScanningStrategy,
    SynchronousFileSystemStrategy: SynchronousFileSystemStrategy,

    // functions
    scanPath: scanPath,
    scanPathSync: scanPathSync
};
