/**
 * @license Copyright (c) 2013, CKSource - Frederico Knabben
 * For licensing, see LICENSE
 */

"use strict";

/* global describe: false, it: false */

var assert = require("assert"),
    Event = require(global.MODULE_ROOT + "/libraries/scripts/Bender/EventDispatcher/Event"),
    EventDispatcher = require(global.MODULE_ROOT + "/libraries/scripts/Bender/EventDispatcher/EventDispatcher");

describe("Bender/EventDispatcher/EventDispatcher", function () {
    var createEventDispatcher; // private, function

    /**
     * @param {array} [supportedEvents]
     * @return {Bender/EventDispatcher/EventDispatcher}
     */
    createEventDispatcher = function (supportedEvents) {
        var eventDispatcher = new EventDispatcher();

        eventDispatcher.getSupportedEvents = function () {
            return supportedEvents;
        };

        return eventDispatcher;
    };

    describe("#addListener", function () {
        it("does not accept anything but function as a listener", function () {
            var eventDispatcher,
                eventName = "test_event_name";

            eventDispatcher = createEventDispatcher([eventName]);
            assert.throws(function () {
                eventDispatcher.addListener(eventName, "not-a-function");
            });
        });

        it("allows listening to supported events only", function () {
            var eventDispatcher = new EventDispatcher(),
                eventName = "assert.event.name";

            assert.throws(function () {
                eventDispatcher.addListener(eventName, function () {});
            }, Error);
        });

        it("allows registering new event listener", function () {
            var eventDispatcher,
                eventName = "assert.event.name",
                listener = function () {};

            eventDispatcher = createEventDispatcher([eventName]);

            assert.ok(!eventDispatcher.hasListeners(eventName));
            assert.ok(!eventDispatcher.hasListener(eventName, listener));

            eventDispatcher.addListener(eventName, listener);

            assert.ok(eventDispatcher.hasListeners(eventName));
            assert.ok(eventDispatcher.hasListener(eventName, listener));
        });

        it("uses event listener priority", function () {
            var eventDispatcher,
                eventA = "assert.event.A",
                eventB = "assert.event.B",
                evt;

            eventDispatcher = createEventDispatcher([eventA, eventB]);

            eventDispatcher.addListener(eventA, function (evt) {
                assert.deepEqual(evt.data, []);
                evt.data.push(1);
            }, 30);

            eventDispatcher.addListener(eventA, function (evt) {
                assert.deepEqual(evt.data, [1, 2, 3]);
                evt.data.push(4);
            }, 10);

            eventDispatcher.addListener(eventA, function (evt) {
                assert.deepEqual(evt.data, [1, 2]);
                evt.data.push(3);
            }, 20);

            eventDispatcher.addListener(eventA, function (evt) {
                assert.deepEqual(evt.data, [1]);
                evt.data.push(2);
            }, 30);

            evt = eventDispatcher.dispatch(eventA, {
                data: []
            });
            assert.deepEqual(evt.data, [1, 2, 3, 4]);
        });
    });

    describe("#addSubscriber", function () {
        it("handles event subscribers", function () {
            var eventDispatcher,
                eventA = "assert.event.A",
                eventB = "assert.event.B";

            eventDispatcher = createEventDispatcher([eventA, eventB]);

            assert.ok(!eventDispatcher.hasListeners(eventA));
            assert.ok(!eventDispatcher.hasListeners(eventB));

            eventDispatcher.addSubscriber({
                getSubscribedEvents: function () {
                    var ret = {};

                    ret[eventA] = function () {};
                    ret[eventB] = function () {};

                    return ret;
                }
            });

            assert.ok(eventDispatcher.hasListeners(eventA));
            assert.ok(eventDispatcher.hasListeners(eventB));
        });
    });

    describe("#dispatch", function () {
        it("dispatches an event", function (done) {
            var eventDispatcher,
                eventName = "test_event_name",
                evt = new Event();

            eventDispatcher = createEventDispatcher([eventName]);
            eventDispatcher.addListener(eventName, function (dispatchedEvent) {
                assert.equal(evt, dispatchedEvent);
                done();
            });

            eventDispatcher.dispatch(eventName, evt);
        });

        it("dispatches an event without data", function (done) {
            var eventDispatcher,
                eventName = "test_event_name";

            eventDispatcher = createEventDispatcher([eventName]);
            eventDispatcher.addListener(eventName, function (evt) {
                assert.ok(evt);
                done();
            });

            eventDispatcher.dispatch(eventName);
        });

        it("cannot dispatch event with empty name (error prone)", function () {
            var eventDispatcher,
                evt = new Event();

            eventDispatcher = createEventDispatcher([void 0]);
            assert.throws(function () {
                eventDispatcher.dispatch(void 0, evt);
            });
        });

        it("does not complain when there are no listeners attached to event", function () {
            var eventDispatcher,
                eventName = "assert.event.name",
                evt = new Event();

            eventDispatcher = createEventDispatcher([eventName]);
            eventDispatcher.dispatch(eventName, evt);
        });

        it("does not support event", function () {
            var eventDispatcher,
                eventName1 = "assert.event.name1",
                eventName2 = "assert.event.name2";

            eventDispatcher = createEventDispatcher([eventName1]);
            assert.throws(function () {
                eventDispatcher.dispatch(eventName2);
            }, Error);
        });

        it("does not dispatch event with event object incompatible data", function () {
            var eventDispatcher,
                eventName = "assert.event.name";

            eventDispatcher = createEventDispatcher([eventName]);
            assert.throws(function () {
                eventDispatcher.dispatch(eventName, "not dispatchable object");
            }, Error);
        });

        it("converts event compatible data to event object", function () {
            var data1 = "value1",
                eventDispatcher,
                eventName = "assert.event.name";

            eventDispatcher = createEventDispatcher([eventName]);
            eventDispatcher.addListener(eventName, function (dispatchedEvent) {
                assert.ok(dispatchedEvent instanceof Event);
                assert.equal(dispatchedEvent.key1, data1);
            });
            eventDispatcher.dispatch(eventName, {
                key1: data1
            });
        });

        it("respects stopped event propagation", function () {
            var eventDispatcher,
                eventName = "assert.event.name",
                evt = new Event();

            eventDispatcher = createEventDispatcher([eventName]);

            eventDispatcher.addListener(eventName, function (dispatchedEvent) {
                assert.equal(evt, dispatchedEvent);
            });

            eventDispatcher.addListener(eventName, function (dispatchedEvent) {
                assert.equal(evt, dispatchedEvent);
                dispatchedEvent.stopPropagation();
            });

            eventDispatcher.addListener(eventName, function (dispatchedEvent) {
                // this assertion should not run
                assert.equal(evt, dispatchedEvent);
            });

            eventDispatcher.dispatch(eventName, evt);
        });
    });

    describe("#getListeners", function () {
        it("does not return listeners with no event specified", function () {
            var eventDispatcher = new EventDispatcher();

            assert.throws(function () {
                eventDispatcher.getListeners();
            }, Error);
        });

        it("returns specific event listeners", function () {
            var eventDispatcher,
                eventA = "assert.event.A",
                listenerA1 = function () {},
                listenerA2 = function () {},
                listeners;

            eventDispatcher = createEventDispatcher([eventA]);

            eventDispatcher.addListener(eventA, listenerA1);
            eventDispatcher.addListener(eventA, listenerA2);

            listeners = eventDispatcher.getListeners(eventA);

            assert.equal(listeners.length, 2);
            assert.notEqual(listeners.indexOf(listenerA1), -1);
            assert.notEqual(listeners.indexOf(listenerA2), -1);
        });
    });

    describe("#isDispatched", function () {
        it("remembers if event was dispatched", function () {
            var eventDispatcher,
                eventA = "assert.event.A",
                eventB = "assert.event.B";

            eventDispatcher = createEventDispatcher([eventA, eventB]);

            assert.ok(!eventDispatcher.isDispatched(eventA));
            assert.ok(!eventDispatcher.isDispatched(eventB));

            eventDispatcher.dispatch(eventA, {});

            assert.ok(eventDispatcher.isDispatched(eventA));
            assert.ok(!eventDispatcher.isDispatched(eventB));

            eventDispatcher.dispatch(eventB, {});

            assert.ok(eventDispatcher.isDispatched(eventA));
            assert.ok(eventDispatcher.isDispatched(eventB));
        });

        it("marks event as dispatched even if propagation stopped", function () {
            var eventDispatcher,
                eventA = "assert.event.A";

            eventDispatcher = createEventDispatcher([eventA]);

            assert.ok(!eventDispatcher.isDispatched(eventA));

            eventDispatcher.addListener(eventA, function (evt) {
                evt.stopPropagation();
            });
            eventDispatcher.dispatch(eventA, {});

            assert.ok(eventDispatcher.isDispatched(eventA));
        });
    });

    describe("#removeListener", function () {
        it("removes event listener", function () {
            var eventDispatcher,
                eventName = "assert.event.name",
                listener = function () {};

            eventDispatcher = createEventDispatcher([eventName]);

            assert.ok(!eventDispatcher.hasListeners(eventName));
            assert.ok(!eventDispatcher.hasListener(eventName, listener));

            eventDispatcher.addListener(eventName, listener);

            assert.ok(eventDispatcher.hasListeners(eventName));
            assert.ok(eventDispatcher.hasListener(eventName, listener));

            eventDispatcher.removeListener(eventName, listener);

            assert.ok(!eventDispatcher.hasListeners(eventName));
            assert.ok(!eventDispatcher.hasListener(eventName, listener));
        });

        it("does not remove listener", function () {
            var eventDispatcher,
                eventName = "assert.event.name",
                listener = function () {};

            eventDispatcher = createEventDispatcher([eventName]);
            assert.throws(function () {
                eventDispatcher.removeListener(eventName, listener);
            }, Error);
        });
    });
});
