/**
 * @license Copyright (c) 2013, CKSource - Frederico Knabben
 * For licensing, see LICENSE
 */

"use strict";

/* global describe: false, it: false */

var assert = require("assert"),
    Event = require(global.MODULE_ROOT + "/libraries/scripts/Bender/EventDispatcher/Event"),
    EventAggregator = require(global.MODULE_ROOT + "/libraries/scripts/Bender/EventDispatcher/EventDispatcherRepository/EventAggregator"),
    EventDispatcher = require(global.MODULE_ROOT + "/libraries/scripts/Bender/EventDispatcher/EventDispatcher");

describe("Bender/EventDispatcher/EventDispatcher/Repository/EventAggregator", function () {

    var createEventDispatcher;

    /**
     * @param {array} supportedEvents
     * @return {Bender/EventDispatcher/EventDispatcher}
     */
    createEventDispatcher = function (supportedEvents) {
        var eventDispatcher = new EventDispatcher();

        eventDispatcher.getSupportedEvents = function () {
            return supportedEvents;
        };

        return eventDispatcher;
    };

    it("aggregates events", function () {
        var eventAggregator = new EventAggregator(),
            eventDispatcherA = createEventDispatcher(["a"]),
            eventDispatcherB = createEventDispatcher(["b"]);

        assert.ok(!eventAggregator.isEventSupported("a"));
        assert.ok(!eventAggregator.isEventSupported("b"));

        eventAggregator.add(eventDispatcherA);

        assert.ok(eventAggregator.isEventSupported("a"));
        assert.ok(!eventAggregator.isEventSupported("b"));

        eventAggregator.add(eventDispatcherB);

        assert.ok(eventAggregator.isEventSupported("a"));
        assert.ok(eventAggregator.isEventSupported("b"));
    });

    it("removes event dispatcher", function () {
        var eventAggregator = new EventAggregator(),
            eventDispatcherAB = createEventDispatcher(["a", "b"]);

        assert.ok(!eventAggregator.isEventSupported("a"));
        assert.ok(!eventAggregator.isEventSupported("b"));

        eventAggregator.add(eventDispatcherAB);

        assert.ok(eventAggregator.isEventSupported("a"));
        assert.ok(eventAggregator.isEventSupported("b"));

        eventAggregator.remove(eventDispatcherAB);

        assert.ok(!eventAggregator.isEventSupported("a"));
        assert.ok(!eventAggregator.isEventSupported("b"));
    });

    it("removes event dispatcher proxy listeners", function () {
        var eventAggregator = new EventAggregator(),
            eventDispatcherAB1 = createEventDispatcher(["a", "b"]),
            eventDispatcherAB2 = createEventDispatcher(["a", "b"]);

        eventAggregator.add(eventDispatcherAB1);
        eventAggregator.add(eventDispatcherAB2);

        eventAggregator.addListener("a", function () {});
        eventAggregator.remove(eventDispatcherAB2);
    });

    it("preserves supported event when at least one attached event dispatcher supports it", function () {
        var eventAggregator = new EventAggregator(),
            eventDispatcherA = createEventDispatcher(["a"]),
            eventDispatcherAB = createEventDispatcher(["a", "b"]);

        assert.ok(!eventAggregator.isEventSupported("a"));
        assert.ok(!eventAggregator.isEventSupported("b"));

        eventAggregator.add(eventDispatcherA);

        assert.ok(eventAggregator.isEventSupported("a"));
        assert.ok(!eventAggregator.isEventSupported("b"));

        eventAggregator.add(eventDispatcherAB);

        assert.ok(eventAggregator.isEventSupported("a"));
        assert.ok(eventAggregator.isEventSupported("b"));

        eventAggregator.remove(eventDispatcherA);

        assert.ok(eventAggregator.isEventSupported("a"));
        assert.ok(eventAggregator.isEventSupported("b"));

        eventAggregator.remove(eventDispatcherAB);

        assert.ok(!eventAggregator.isEventSupported("a"));
        assert.ok(!eventAggregator.isEventSupported("b"));
    });

    it("notifies event listeners", function (done) {
        var eventAggregator = new EventAggregator(),
            eventDispatcherA = createEventDispatcher(["a"]),
            eventDispatcherB = createEventDispatcher(["b"]),
            evt;

        eventAggregator.add(eventDispatcherA);
        eventAggregator.add(eventDispatcherB);

        evt = new Event();

        eventAggregator.addListener("a", function (dispatchedEvent) {
            assert.strictEqual(dispatchedEvent, evt);
        });

        eventAggregator.addListener("b", function (dispatchedEvent) {
            assert.strictEqual(dispatchedEvent, evt);

            done();
        });

        eventDispatcherA.dispatch("a", evt);
        eventDispatcherB.dispatch("b", evt);
    });

    it("notifies listeners when at least one supported event dispatcher is attached", function (done) {
        var eventAggregator = new EventAggregator(),
            eventDispatcherA = createEventDispatcher(["a"]),
            eventDispatcherAB = createEventDispatcher(["a", "b"]),
            evt;

        eventAggregator.add(eventDispatcherA);
        eventAggregator.add(eventDispatcherAB);

        evt = new Event();

        eventAggregator.addListener("a", function (dispatchedEvent) {
            assert.strictEqual(dispatchedEvent, evt);
        });

        eventAggregator.addListener("b", function (dispatchedEvent) {
            assert.strictEqual(dispatchedEvent, evt);

            done();
        });

        eventDispatcherA.dispatch("a", evt);
        eventDispatcherAB.dispatch("a", evt);
        eventDispatcherAB.dispatch("b", evt);
    });

});
