/**
 * @license Copyright (c) 2013, CKSource - Frederico Knabben
 * For licensing, see LICENSE
 */

"use strict";

/* global define: false */

(function (factory) {

    if ("function" === typeof define) {
        define([
            "configuration-processor/configuration-processor",
            "event-dispatcher/Event",
            "event-dispatcher/ListenerLinkedList"
        ], factory);
    } else {
        module.exports = factory.apply(null, [
            require("configuration-processor"),
            require(__dirname + "/Event"),
            require(__dirname + "/ListenerLinkedList")
        ]);
    }

}(function (configurationProcessor, Event, ListenerLinkedList) {

    var assertEventDispatcherSupportsEvent, // private, function
        config = configurationProcessor.configurationProcessor,
        priorityComparator, // private, function
        EventDispatcher; // constructor, function

    /**
     * @param {Bender/EventDispatcher/EventDispatcher} eventDispatcher
     * @param {string} eventName
     * @return {void}
     * @throws {Error} if event dispatcher does not support given event
     */
    assertEventDispatcherSupportsEvent = function (eventDispatcher, eventName) {
        var message,
            supportedEvents;

        if (eventDispatcher.isEventSupported(eventName)) {
            return;
        }

        if (eventName) {
            message = "Event dispatcher does not support event: " + eventName + ".";
        } else {
            message = "Empty event name given: " + eventName + ".";
        }

        supportedEvents = eventDispatcher.getSupportedEvents();
        if (supportedEvents.length) {
            message += " Supported events are: '" + supportedEvents.join("', '") + "'.";
        }

        throw new Error(message);
    };

    /**
     * @access private
     * @param {Bender/EventDispatcher/ListenerLinkedListNode} listNodeA
     * @param {Bender/EventDispatcher/ListenerLinkedListNode} listNodeB
     * @return {number}
     */
    priorityComparator = function (listNodeA, listNodeB) {
        return listNodeB.priority - listNodeA.priority;
    };

    /**
     * @constructor
     */
    EventDispatcher = function () {
        this.dispatchLog = {};
        this.listeners = {};
    };

    /**
     * @param {string} eventName
     * @param {function} listener
     * @param {int} [priority] 0 by default
     * @return {void}
     * @throws {TypeError} if given listener is not a function
     */
    EventDispatcher.prototype.addListener = function (eventName, listener, priority) {
        listener = config.assertFunction(listener);
        assertEventDispatcherSupportsEvent(this, eventName);

        if (priority === void 0) {
            priority = 0;
        }

        if (!this.listeners.hasOwnProperty(eventName)) {
            this.listeners[eventName] = this.encapsulateListenerList();
        }

        this.listeners[eventName].add(listener).priority = priority;
        this.listeners[eventName].sort(priorityComparator);
    };

    /**
     * @param {object} subscriber
     * @return {void}
     */
    EventDispatcher.prototype.addSubscriber = function (subscriber) {
        var subscribedEvent,
            subscribedEvents = subscriber.getSubscribedEvents(this);

        for (subscribedEvent in subscribedEvents) {
            if (subscribedEvents.hasOwnProperty(subscribedEvent)) {
                this.addListener(subscribedEvent, subscribedEvents[subscribedEvent]);
            }
        }
    };

    /**
     * @param {string} eventName
     * @param {Bender/EventDispatcher/Event|object} evt (optional)
     * @return {Bender/EventDispatcher/Event} dispatched event
     */
    EventDispatcher.prototype.dispatch = function (eventName, evt) {
        assertEventDispatcherSupportsEvent(this, eventName);

        var i,
            listeners;

        if (!(evt instanceof Event)) {
            evt = new Event(evt);
        }

        this.dispatchLog[eventName] = new Date();

        evt.dispatcher = this;
        if (this.hasListeners(eventName)) {
            listeners = this.getListeners(eventName).list;
            for (i = 0; i < listeners.length; i += 1) {
                listeners[i].value(evt);
                if (evt.isPropagationStopped) {
                    return evt;
                }
            }
        }

        return evt;
    };

    /**
     * @param {array} listeners
     * @return {Bender/EventDispatcher/ListenerLinkedList}
     */
    EventDispatcher.prototype.encapsulateListenerList = function (listeners) {
        return new ListenerLinkedList(listeners);
    };

    /**
     * @return {array}
     */
    EventDispatcher.prototype.getSupportedEvents = function () {
        return [];
    };

    /**
     * @param {string} eventName
     * @return {Bender/EventDispatcher/ListenerLinkedList}
     * @throws {Error} if no listeners are registered
     */
    EventDispatcher.prototype.getListeners = function (eventName) {
        if (!this.hasListeners(eventName)) {
            throw new Error("There are no listeners assigned to: " + eventName);
        }

        return this.listeners[eventName];
    };

    /**
     * @param {string} eventName
     * @param {function} listener
     * @return {bool}
     */
    EventDispatcher.prototype.hasListener = function (eventName, listener) {
        listener = config.assertFunction(listener);
        if (!this.hasListeners(eventName)) {
            return false;
        }

        return this.listeners[eventName].has(listener);
    };

    /**
     * @param {string} eventName
     * @return {bool}
     */
    EventDispatcher.prototype.hasListeners = function (eventName) {
        return this.listeners.hasOwnProperty(eventName) && this.listeners[eventName].length;
    };

    /**
     * @param {string} eventName
     * @return {bool}
     */
    EventDispatcher.prototype.isDispatched = function (eventName) {
        assertEventDispatcherSupportsEvent(this, eventName);

        return !!this.dispatchLog[eventName];
    };

    /**
     * @return {bool}
     */
    EventDispatcher.prototype.isEventSupported = function (eventName) {
        return eventName && (-1 !== this.getSupportedEvents().indexOf(eventName));
    };

    /**
     * @param {string} eventName
     * @param {function} listener
     * @return {void}
     * @throws {Error} if listener is not assigned to this dispatcher
     */
    EventDispatcher.prototype.removeListener = function (eventName, listener) {
        if (!this.hasListener(eventName, listener)) {
            throw new Error("Given listener is not assigned to event: " + eventName);
        }

        this.listeners[eventName].remove(listener);
    };

    return EventDispatcher;

}));
