# Event Dispatcher

Alongside standard event emitting mechanism, this library provides
some additional utilities like event aggregator and generic event dispatching
repository.

This is an **AMD** library. To use it with Node.js and other CommonJS modules
based environments I recommend
[jsloader/jsloader](https://github.com/jsloader/jsloader).

This library is partially modeled after
[symfony/EventDispatcher](http://github.com/symfony/EventDispatcher).

## Examples

### Basic usage

#### Dispatching events

```JavaScript
require([
    "Bender/EventDispatcher/Event",
    "Bender/EventDispatcher/EventDispatcher"
], function (Event, EventDispatcher) {
    var eventDispatcher = new EventDispatcher();

    eventDispatcher.getSupportedEvents = function () {
        return ["a"];
    };

    eventDispatcher.addListener("a", function (evt) {
        console.log("hello %s!", evt.data.place); // hello world
    });

    eventDispatcher.dispatch("a", new Event({ place: "world" }));
});
```

#### Aggregating event dispatchers

```JavaScript
var auto = require("jsloader"),
    eventAggregator,
    eventDispatcherA,
    eventDispatcherB,
    Event = auto.load("Bender/EventDispatcher/Event"),
    EventAggregator = auto.load("Bender/EventDispatcher/EventAggregator"),
    EventDispatcher = auto.load("Bender/EventDispatcher/EventDispatcher");

eventAggregator = new EventAggregator();

eventDispatcherA.getSupportedEvents = function () {
    return ["a"];
};

eventDispatcherB.getSupportedEvents = function () {
    return ["b"];
};

eventAggregator.add(eventDispatcherA);
eventAggregator.add(eventDispatcherB);

eventAggregator.addListener("a", function (evt) {
    console.log("hello %s!", evt.data.place); // hello world
});

eventAggregator.addListener("b", function (evt) {
    console.log("hello %s!", evt.data.place); // hello planet
});

eventDispatcherA.dispatch("a", new Event({ place: "world" }));
eventDispatcherB.dispatch("b", new Event({ place: "planet" }));
```

### Advanced usage

#### Exposing single module entry point using two combined modules

```JavaScript
// module/a.js

var eventAggregator,
    eventDispatcherAB = require("foo").eventDispatcherAB,
    eventDispatcherC = require("bar").eventDispatcherC;

eventAggregator = new EventAggregator();

eventAggregator.add(eventDispatcherAB);
eventAggregator.add(eventDispatcherC);

module.exports = eventAggregator;
```

```JavaScript
// module/b.js

var a = require("a");

// eventDispatcherAB is going to handle this
a.addListener("a", function () {});

// eventDispatcherAB is going to handle this
a.addListener("b", function () {});

// eventDispatcherC is going to handle this
a.addListener("c", function () {});
```

#### Generic repository based on event dispatcher

```JavaScript
define(["Bender/EventDispatcher/EventDispatcher/Repository"], function (Repository) {
    var repository = new Repository();

    repository.addListener(Repository.EVENT_ITEM_ADDED, function (evt) {
        var item = evt.data.item,
            repository = evt.data.repository;

        console.log("added: " + item);

        if (("world" === item) && 2148 !== (new Date().getFullYear())) {
            // this one should not be here for some reason
            repository.remove(item);
        }
    });

    repository.addListener(Repository.EVENT_ITEM_REMOVED, function (evt) {
        console.log("removed: " + evt.data.item);
    });

    repository.add("hello"); // added: hello
    repository.add("world"); // added: world, removed: world
});
```

## Semantic Versioning

This repository follows [Semantic Versioning](http://semver.org/) convention.
