/**
 * @license Copyright (c) 2013, Cheng Lou (github.com/chenglou)
 * @license Copyright (c) 2013, CKSource - Frederico Knabben
 * For licensing, see LICENSE
 */

"use strict";

/* global describe: false, it: false */

var assert = require("assert"),
    Trie = require(global.MODULE_ROOT + "/libraries/scripts/Bender/DataContainer/Trie");

describe("Bender/DataContainer/Trie", function () {
    var empty,
        fill;

    fill = function (trie) {
        var initialWordCount;
        initialWordCount = trie.size;
        trie.add("he");
        trie.add("hello");
        trie.add("ello");
        trie.add("bell");
        trie.add("hell");
        trie.add("bear");
        trie.add("beer");
        trie.add("z");
        trie.add("za");
        trie.add("zz");
        assert.strictEqual(trie.size, initialWordCount + 10);
    };

    empty = function (trie) {
        var initialWordCount;
        initialWordCount = trie.size;
        trie.remove("he");
        trie.remove("hello");
        trie.remove("ello");
        trie.remove("bell");
        trie.remove("hell");
        trie.remove("bear");
        trie.remove("beer");
        trie.remove("z");
        trie.remove("za");
        trie.remove("zz");
        assert.strictEqual(trie.size, Math.max(initialWordCount - 10, 0));
    };

    describe("Initialization with an array", function () {
        var trie;
        trie = new Trie(["hello", "he", "hell", "beer", "apple"]);
        it("should have formed the trie correctly", function () {
            assert.ok(trie.has("hello"));
            assert.ok(trie.has("hell"));
            assert.ok(trie.has("he"));
            assert.ok(trie.has("beer"));
            assert.ok(trie.has("apple"));
        });
    });

    describe("#add", function () {
        var trie = new Trie();

        it("should have an empty word  count at the beginning", function () {
            assert.strictEqual(trie.size, 0);
        });
        it("should return nothing when the trie's empty", function () {
            assert.ok(!trie.has(""));
            assert.ok(!trie.has("beer"));
        });
        it("should return the word added", function () {
            assert.strictEqual(trie.add("he"), "he");
            assert.strictEqual(trie.add("hello"), "hello");
            assert.strictEqual(trie.add("hell"), "hell");
            assert.strictEqual(trie.add("beer"), "beer");
            assert.strictEqual(trie.size, 4);
        });
        it("should find the words just added, and nothing but", function () {
            assert.ok(trie.has("hello"));
            assert.ok(trie.has("hell"));
            assert.ok(trie.has("he"));
            assert.ok(trie.has("beer"));
            assert.ok(!trie.has("bee"));
        });
        it("should not allow adding anything but string", function () {
            assert.throws(function () {
                trie.add(null);
            });
        });
        it("should add an empty string as a valid word", function () {
            assert.ok(!trie.has(""));
            assert.strictEqual(trie.add(""), "");
            assert.ok(trie.has(""));
        });
        it("should not add the same word more than once", function () {
            fill(trie);
            assert.ok(trie.has(""));
            assert.ok(trie.has("he"));
            assert.ok(trie.has("hello"));
            assert.ok(trie.has("hell"));
            assert.ok(trie.has("beer"));
            assert.ok(trie.has("bear"));
            assert.ok(trie.has("z"));
            assert.ok(trie.has("za"));
            assert.ok(trie.has("zz"));
        });
    });

    describe("#longestPrefixOf", function () {
        var trie = new Trie();

        it("should return an empty string if the word's not found", function () {
            assert.strictEqual(trie.longestPrefixOf(""), "");
            assert.strictEqual(trie.longestPrefixOf("beer"), "");
        });
        it("should return the correct prefix otherwise", function () {
            fill(trie);
            assert.strictEqual(trie.longestPrefixOf("hel"), "hel");
            assert.strictEqual(trie.longestPrefixOf("hellha"), "hell");
            assert.strictEqual(trie.longestPrefixOf("beer cup"), "beer");
            assert.strictEqual(trie.longestPrefixOf("h"), "h");
            assert.strictEqual(trie.longestPrefixOf("beers"), "beer");
            assert.strictEqual(trie.longestPrefixOf(""), "");
            assert.strictEqual(trie.longestPrefixOf("a"), "");
        });
    });

    describe("#remove", function () {
        var trie = new Trie();

        it("should return undefined if trie's empty", function () {
            assert.ok(!trie.remove("hello"));
            assert.ok(!trie.remove("h"));
            assert.ok(!trie.remove(""));
        });
        it("should return undefined if some letters of the word aren't there", function () {
            fill(trie);
            assert.ok(!trie.remove("hel"));
            assert.ok(!trie.remove("b"));
            assert.ok(!trie.remove("zzz"));
            assert.strictEqual(trie.size, 10);
        });
        it("should be case-sensitive", function () {
            assert.ok(!trie.remove("He"));
        });
        it("should return the word removed", function () {
            assert.strictEqual(trie.remove("he"), "he");
            assert.ok(!trie.has("he"));
            assert.strictEqual(trie.remove("zz"), "zz");
            assert.ok(!trie.has("zz"));
            assert.strictEqual(trie.size, 8);
        });
        it("should not have removed letters if other words are using them", function () {
            assert.ok(trie.has("hello"));
            assert.ok(trie.has("hell"));
            assert.ok(trie.has("ello"));
            assert.ok(trie.has("bell"));
            assert.ok(trie.has("bear"));
            assert.ok(trie.has("beer"));
            assert.ok(trie.has("z"));
            assert.ok(trie.has("za"));
        });
        it("should remove the letters if no word is using them anymore", function () {
            trie.remove("hello");
            assert.ok(trie.has("hell"));
            trie.remove("hell");
            assert.deepEqual(trie.wordsWithPrefix("h"), []);
        });
        it("should affect other branches", function () {
            assert.ok(trie.has("ello"));
            assert.ok(trie.has("bell"));
        });
        it("should conserve empty string if it exists", function () {
            trie.add("");
            trie.remove("z");
            assert.ok(trie.has(""));
            trie.remove("za");
            assert.ok(trie.has(""));
        });
        it("should remove empty string if it was a word", function () {
            assert.strictEqual(trie.remove(""), "");
            assert.ok(!trie.has(""));
        });
        it("should leave an empty trie after removing everything", function () {
            empty(trie);
        });
    });

    describe("#wordsWithPrefix", function () {
        var trie = new Trie();

        it("should return an empty array if nothing's found", function () {
            assert.deepEqual(trie.wordsWithPrefix("asd"), []);
            assert.deepEqual(trie.wordsWithPrefix(""), []);
        });
        it("should return every word if an empty string is passed", function () {
            fill(trie);
            assert.deepEqual(trie.wordsWithPrefix(""), ["z", "he", "za", "zz", "hell", "ello", "bell", "bear", "beer", "hello"]);
        });
        it("should return the same array, plus the empty string, if it was added", function () {
            trie.add("");
            assert.deepEqual(trie.wordsWithPrefix(""), ["", "z", "he", "za", "zz", "hell", "ello", "bell", "bear", "beer", "hello"]);
        });
        it("should include the word itself", function () {
            assert.deepEqual(trie.wordsWithPrefix("hell"), ["hell", "hello"]);
        });
    });
});
