/**
 * @license Copyright (c) 2013, CKSource - Frederico Knabben
 * For licensing, see LICENSE
 */

"use strict";

/* global describe: false, it: false */

var assert = require("assert"),
    Repository = require(global.MODULE_ROOT + "/libraries/scripts/Bender/DataContainer/Repository");

describe("Bender/DataContainer/Repository", function () {
    it("creates repository with initial collection", function () {
        var item1 = "test1",
            item2 = "test2",
            item3 = "test3",
            repository = new Repository([item1, item2, item3]);

        assert.equal(3, repository.length);

        assert.ok(repository.has(item1));
        assert.ok(repository.has(item2));
        assert.ok(repository.has(item3));
    });

    describe("#add", function () {
        it("adds item", function () {
            var item = "test item",
                repository = new Repository();

            assert.equal(0, repository.length);

            repository.add(item);

            assert.equal(1, repository.length);
        });

        it("does not allow item", function () {
            var item = "test item",
                repository = new Repository();

            repository.isItemAllowed = function (item) {
                return false;
            };

            assert.throws(function () {
                repository.add(item);
            }, Error);
        });
    });

    describe("#at", function () {
        it("fails when index is out of repository range", function () {
            var repository = new Repository([1, 2, 3]);

            assert.throws(function () {
                repository.at(repository.length);
            });
        });

        it("fetches item from given position", function () {
            var repository = new Repository([1, 2, 3]);

            assert.strictEqual(2, repository.at(1));
        });
    });

    describe("#get", function () {
        it("fails when repository cannot get item", function () {
            var repository = new Repository([1, 2, 3]);

            assert.throws(function () {
                repository.get(4);
            });
        });

        it("fetches item from repository", function () {
            var repository = new Repository([1, 2, 3]);

            assert.strictEqual(2, repository.get(2));
        });
    });

    describe("#indexOf", function () {
        it("finds item index", function () {
            var repository,
                o = {};

            repository = new Repository([1, 2, o]);
            assert.strictEqual(repository.at(repository.indexOf(o)), o);
        });
    });

    describe("#remove", function () {
        it("does not remove item that is not a part of repository", function () {
            var item = "test item",
                repository = new Repository();

            assert.throws(function () {
                repository.remove(item);
            }, Error);
        });

        it("removes item", function () {
            var item = "test item",
                repository = new Repository();

            assert.equal(0, repository.length);

            repository.add(item);

            assert.equal(1, repository.length);

            repository.remove(item);

            assert.equal(0, repository.length);
        });
    });

    describe("#shift", function () {
        it("removes first element from repository", function () {
            var repository;

            repository = new Repository();
            repository.add(1);
            repository.add(2);

            assert.equal(repository.length, 2);
            assert.strictEqual(repository.shift(), 1);
            assert.equal(repository.length, 1);
        });
    });

    describe("#unshift", function () {
        it("fails when item cannot be added", function () {
            var repository;

            repository = new Repository();
            repository.canAdd = function (item) {
                return item > 3;
            };

            assert.throws(function () {
                repository.unshift(1);
            });
        });

        it("prepends element to repository", function () {
            var repository;

            repository = new Repository();
            repository.unshift(1);
            repository.unshift(2);

            assert.strictEqual(repository.indexOf(2), 0);
        });
    });
});
