/**
 * @license Copyright (c) 2013, CKSource - Frederico Knabben
 * For licensing, see LICENSE
 */

"use strict";

/* global describe: false, it: false */

var assert = require("assert"),
    LinkedList = require(global.MODULE_ROOT + "/libraries/scripts/Bender/DataContainer/Repository/LinkedList");

describe("Bender/DataContainer/LinkedList", function () {
    it("can be initialized with an array", function () {
        var linkedList = new LinkedList([1, 2, 3]);

        assert.strictEqual(linkedList.length, 3);
        assert.strictEqual(linkedList.head.value, 1);
        assert.strictEqual(linkedList.head.next.value, 2);
        assert.strictEqual(linkedList.head.next.next.value, 3);
        assert.strictEqual(linkedList.head.next.next, linkedList.last);
    });

    describe("#add", function () {
        it("adds item to the end of the list", function () {
            var linkedList = new LinkedList();

            linkedList.add("hello");

            assert.strictEqual(linkedList.head.value, "hello");
        });

        it("preserves items chain", function () {
            var linkedList = new LinkedList();

            linkedList.add("a");
            linkedList.add("b");
            linkedList.add("c");

            assert.equal(linkedList.length, 3);
            assert.strictEqual(linkedList.head.value, "a");
            assert.strictEqual(linkedList.head.next.value, "b");
            assert.strictEqual(linkedList.head.next.next.value, "c");
        });
    });

    describe("#at", function () {
        it("returns node at given index", function () {
            var linkedList = new LinkedList(),
                nodeA,
                nodeC;

            nodeA = linkedList.add("a");
            nodeC = linkedList.add("c");

            assert.strictEqual(linkedList.at(0), nodeA);
            assert.strictEqual(linkedList.at(1), nodeC);
        });
    });

    describe("#get", function () {
        it("returns node with given value", function () {
            var linkedList = new LinkedList(),
                nodeA,
                nodeC;

            nodeA = linkedList.add("a");
            nodeC = linkedList.add("c");

            assert.strictEqual(linkedList.get("a"), nodeA);
            assert.strictEqual(linkedList.get("c"), nodeC);
        });
    });

    describe("#has", function () {
        it("answers negatively", function () {
            var linkedList = new LinkedList();

            linkedList.add("a");
            linkedList.add("c");

            assert.ok(!linkedList.has("b"));
        });

        it("answers positively", function () {
            var linkedList = new LinkedList();

            linkedList.add("a");
            linkedList.add("c");

            assert.ok(linkedList.has("a"));
        });
    });

    describe("#head", function () {
        it("makes head and last the same element", function () {
            var linkedList = new LinkedList();

            linkedList.add("a");

            assert.strictEqual(linkedList.head, linkedList.last);
        });
    });

    describe("#remove", function () {
        it("preserves items chain", function () {
            var linkedList = new LinkedList();

            linkedList.add("a");
            linkedList.add("b");
            linkedList.add("c");

            linkedList.remove("b");

            assert.equal(linkedList.length, 2);
            assert.strictEqual(linkedList.head.value, "a");
            assert.strictEqual(linkedList.head.next.value, "c");
        });

        it("removes head", function () {
            var linkedList = new LinkedList();

            linkedList.add("a");
            linkedList.add("b");

            linkedList.remove(linkedList.head.value);

            assert.ok(!linkedList.head.next);
            assert.ok(!linkedList.head.previous);

            assert.ok(!linkedList.last.next);
            assert.ok(!linkedList.last.previous);

            assert.strictEqual(linkedList.head, linkedList.last);
        });

        it("removes last", function () {
            var linkedList = new LinkedList();

            linkedList.add("a");
            linkedList.add("b");

            linkedList.remove(linkedList.last.value);

            assert.ok(!linkedList.head.next);
            assert.ok(!linkedList.head.previous);

            assert.ok(!linkedList.last.next);
            assert.ok(!linkedList.last.previous);

            assert.strictEqual(linkedList.head, linkedList.last);
        });

        it("removes the only node", function () {
            var linkedList = new LinkedList();

            linkedList.add("a");
            assert.equal(linkedList.length, 1);

            linkedList.remove("a");
            assert.equal(linkedList.length, 0);
        });
    });

    describe("#sort", function () {
        it("preserves node instances", function () {
            var linkedList = new LinkedList(),
                previousHead,
                previousLast;

            linkedList.add("b");
            linkedList.add("a");

            assert.equal(linkedList.length, 2);

            assert.strictEqual(linkedList.head.value, "b");
            previousHead = linkedList.head;

            assert.strictEqual(linkedList.head.next.value, "a");
            assert.strictEqual(linkedList.head.next, linkedList.last);
            previousLast = linkedList.last;

            linkedList.sort();

            assert.equal(linkedList.length, 2);
            assert.strictEqual(linkedList.head.value, "a");
            assert.strictEqual(linkedList.head.next.value, "b");
            assert.strictEqual(linkedList.head.next, linkedList.last);

            assert.strictEqual(previousHead, linkedList.last);
            assert.strictEqual(previousLast, linkedList.head);
        });

        it("sorts list items", function () {
            var linkedList = new LinkedList();

            linkedList.add("b");
            linkedList.add("d");
            linkedList.add("c");
            linkedList.add("a");

            assert.equal(linkedList.length, 4);
            assert.strictEqual(linkedList.head.value, "b");
            assert.strictEqual(linkedList.head.next.value, "d");
            assert.strictEqual(linkedList.head.next.next.value, "c");
            assert.strictEqual(linkedList.head.next.next.next.value, "a");
            assert.strictEqual(linkedList.head.next.next.next, linkedList.last);

            linkedList.sort();

            assert.equal(linkedList.length, 4);
            assert.strictEqual(linkedList.head.value, "a");
            assert.strictEqual(linkedList.head.next.value, "b");
            assert.strictEqual(linkedList.head.next.next.value, "c");
            assert.strictEqual(linkedList.head.next.next.next.value, "d");
            assert.strictEqual(linkedList.head.next.next.next, linkedList.last);
        });

        it("uses custom sort comparator", function () {
            var linkedList = new LinkedList();

            linkedList.add("b");
            linkedList.add("d");
            linkedList.add("c");
            linkedList.add("a");

            assert.equal(linkedList.length, 4);
            assert.strictEqual(linkedList.head.value, "b");
            assert.strictEqual(linkedList.head.next.value, "d");
            assert.strictEqual(linkedList.head.next.next.value, "c");
            assert.strictEqual(linkedList.head.next.next.next.value, "a");
            assert.strictEqual(linkedList.head.next.next.next, linkedList.last);

            linkedList.sort(function (itemA, itemB) {
                if (itemA.value > itemB.value) {
                    return -1;
                }
                if (itemA.value < itemB.value) {
                    return 1;
                }
                return 0;
            });

            assert.equal(linkedList.length, 4);
            assert.strictEqual(linkedList.head.value, "d");
            assert.strictEqual(linkedList.head.next.value, "c");
            assert.strictEqual(linkedList.head.next.next.value, "b");
            assert.strictEqual(linkedList.head.next.next.next.value, "a");
            assert.strictEqual(linkedList.head.next.next.next, linkedList.last);
        });
    });

    describe("#unshift", function () {
        it("adds item to the beginning of the list", function () {
            var linkedList = new LinkedList();

            linkedList.unshift("hello");

            assert.strictEqual(linkedList.head.value, "hello");
        });

        it("preserves items chain", function () {
            var linkedList = new LinkedList();

            linkedList.unshift("a");
            linkedList.unshift("b");
            linkedList.unshift("c");

            assert.equal(linkedList.length, 3);
            assert.strictEqual(linkedList.head.value, "c");
            assert.strictEqual(linkedList.head.next.value, "b");
            assert.strictEqual(linkedList.head.next.next.value, "a");
        });
    });
});
