/**
 * @license Copyright (c) 2013, Cheng Lou (github.com/chenglou)
 * @license Copyright (c) 2013, CKSource - Frederico Knabben
 * For licensing, see LICENSE
 */

"use strict";

/* global describe: false, it: false */

var assert = require("assert"),
    DataMap = require(global.MODULE_ROOT + "/libraries/scripts/Bender/DataContainer/Map");

describe("Bender/DataContainer/Map", function () {
    var fixtureFunction, // private, function
        fixtureObjectA,
        fixtureObjectB;

    /**
     * @access private
     * @return {string}
     */
    fixtureFunction = function () {
        return "hello";
    };

    fixtureObjectA = {
        a: 10
    };

    fixtureObjectB = {
        b: 10
    };

    it("can be constructed with initial map", function () {
        var map;

        map = new DataMap({
            a: "b",
            c: "d"
        });

        assert.ok(map.has("a"));
        assert.ok(map.has("c"));
    });

    describe("#filter", function () {
        it("filters map elements", function () {
            var filtered = new DataMap(),
                map;

            map = new DataMap({
                a: 1,
                b: 2,
                c: 3
            });

            assert.strictEqual(filtered.length, 0);
            map.filter(filtered, function (key, value) {
                return value > 1;
            });

            assert.deepEqual(filtered.keys(), ["b", "c"]);
        });

        it("filters map with custom context", function (done) {
            var context = {},
                filtered = new DataMap(),
                map;

            map = new DataMap({
                a: 1
            });

            assert.strictEqual(filtered.length, 0);
            map.filter(filtered, function (key, value) {
                assert.strictEqual(this, context);
                done();
            }, context);
        });
    });

    describe("#forEach", function () {
        it("iterates over every map element", function () {
            var map;

            map = new DataMap({
                a: {
                    initialIndex: "a"
                },
                b: {
                    initialIndex: "b"
                },
                c: {
                    initialIndex: "c"
                }
            });

            map.forEach(function (key, value) {
                assert.strictEqual(key, value.initialIndex);
                value.visitedIndex = key;
            });

            assert.strictEqual(map.get("a").visitedIndex, "a");
            assert.strictEqual(map.get("b").visitedIndex, "b");
            assert.strictEqual(map.get("c").visitedIndex, "c");
        });
    });

    describe("#pick", function () {
        it("filters map elements and removes them from source", function () {
            var picked = new DataMap(),
                map;

            map = new DataMap({
                a: 1,
                b: 2,
                c: 3
            });

            map.pick(picked, function (key, value) {
                return value > 1;
            });

            assert.deepEqual(map.keys(), ["a"]);
            assert.deepEqual(picked.keys(), ["b", "c"]);
        });

        it("picks elements with custom context", function (done) {
            var context = {},
                filtered = new DataMap(),
                map;

            map = new DataMap({
                a: 1
            });

            assert.strictEqual(filtered.length, 0);
            map.pick(filtered, function (key, value) {
                assert.strictEqual(this, context);
                done();
            }, context);
        });
    });

    describe("#remove", function () {
        var arr, date1, date2, map, obj;
        map = new DataMap();
        it("should throw error for deleting nonexistent keys", function () {
            assert.throws(function () {
                map.remove(5);
            });
        });
        it("shouldn't decrease the size count after fail deletes", function () {
            assert.strictEqual(map.length, 0);
        });
        arr = [
            1, 2, {
                a: "hello"
            },
            [3, 4, 5]
        ];
        obj = {
            a: "hi",
            b: {},
            c: [1, 2],
            d: arr
        };
        date1 = new Date();
        date2 = new Date();
        it("should return true after deleting a valid key", function () {
            map.set(5, "number 6");
            map.set("5", "string 6");
            map.set(void 0, [3, 2]);
            map.set(null, fixtureObjectB);
            map.set(true, "okay");
            map.set(/asd/, true);
            map.set(fixtureFunction, 10);
            map.remove(5);
            map.remove("5");
            map.remove(void 0);
            map.remove(null);
            map.remove(true);
            map.remove(/asd/);
            map.remove(fixtureFunction);
        });
        it("should have updated the size", function () {
            assert.strictEqual(map.length, 0);
        });
        it("shouldn't find anything after emptying a map", function () {
            assert.ok(!map.has(5));
            assert.ok(!map.has("5"));
            assert.ok(!map.has(void 0));
            assert.ok(!map.has(null));
            assert.ok(!map.has(true));
            assert.ok(!map.has(/asd/));
            assert.ok(!map.has(fixtureFunction));
        });
        it("should keep the count correct after removing special keys", function () {
            assert.strictEqual(map.length, 0);
        });
    });

    describe("#set", function () {
        var arr,
            date1 = new Date(),
            date2 = new Date(),
            map = new DataMap(),
            map2 = new DataMap(),
            obj;

        arr = [
            1, 2, {
                a: "hello"
            },
            [3, 4, 5]
        ];
        obj = {
            a: "hi",
            b: {},
            c: [1, 2],
            d: arr
        };

        it("can reject specific keys", function () {
            var map = new DataMap();

            map.isKeyAllowed = function (key) {
                return key < 5;
            };

            assert.throws(function () {
                map.set(6, "six");
            });
        });

        it("can reject specific values", function () {
            var map = new DataMap();

            map.isItemAllowed = function (item) {
                return "string" !== typeof item;
            };

            assert.throws(function () {
                map.set(6, "six");
            });
        });

        it("should have an empty size initially", function () {
            assert.strictEqual(map.length, 0);
        });

        it("should have nothing when map's empty", function () {
            assert.ok(!map.has(5));
            assert.ok(!map.has(void 0));
            assert.ok(!map.has(null));
            assert.ok(!map.has(0));
            assert.ok(!map.has([]));
            assert.ok(!map.has(fixtureFunction));
            assert.ok(!map.has("5"));
        });

        it("should put the key and its value into the map", function () {
            map.set(5, "number 5");
            assert.ok(map.has(5));
            assert.strictEqual(map.get(5), "number 5");

            map.set("5", "string 5");
            assert.ok(map.has("5"));
            assert.strictEqual(map.get("5"), "string 5");

            map.set(void 0, [1, 2, 3]);
            assert.ok(map.has(void 0));
            assert.deepEqual(map.get(void 0), [1, 2, 3]);

            map.set(null, fixtureObjectA);
            assert.ok(map.has(null));
            assert.deepEqual(map.get(null), fixtureObjectA);

            map.set(true, "ok");
            assert.ok(map.has(true));
            assert.strictEqual(map.get(true), "ok");

            map.set(/asd/, false);
            assert.ok(map.has(/asd/));
            assert.strictEqual(map.get(/asd/), false);

            map.set(fixtureFunction, 99);
            assert.ok(map.has(fixtureFunction));
            assert.strictEqual(map.get(fixtureFunction), 99);

            map.set(arr, "array");
            assert.ok(map.has(arr, "time"));
            assert.strictEqual(map.get(arr, "time"), "array");

            map.set(obj, "obj");
            assert.ok(map.has(obj, "time"));
            assert.strictEqual(map.get(obj, "time"), "obj");
        });

        it("should keep track of map size", function () {
            assert.strictEqual(map.length, 9);
        });

        it("should override previous value", function () {
            map.set(5, "number 6");
            map.set("5", "string 6");
            map.set(void 0, [3, 2]);
            map.set(null, fixtureObjectB);
            map.set(true, "okay");
            map.set(/asd/, true);
            map.set(fixtureFunction, 10);
            assert.strictEqual(map.get(5), "number 6");
            assert.strictEqual(map.get("5"), "string 6");
            assert.deepEqual(map.get(void 0), [3, 2]);
            assert.deepEqual(map.get(null), fixtureObjectB);
            assert.strictEqual(map.get(true), "okay");
            assert.strictEqual(map.get(/asd/), true);
            assert.strictEqual(map.get(fixtureFunction), 10);
        });

        it("shouldn't have changed the map's size", function () {
            assert.strictEqual(map.length, 9);
        });

        it("should throw error if the key's not found", function () {
            assert.throws(function () {
                map.get(6);
            });
        });

        it("should store two Date objects distinctively", function () {
            map.set(date1, "date 1");
            map.set(date2, "date 2");
            assert.strictEqual(map.get(date1), "date 1");
            assert.strictEqual(map.get(date2), "date 2");
            assert.ok(!map.has(new Date()));
            assert.ok(!map.has(new Date()));
        });

        it("should store a id on the key that's unique to each map", function () {
            assert.ok(!map2.has(date1));
            map2.set(date1, "date 1 on map 2");
            assert.strictEqual(map.get(date1), "date 1");
            assert.strictEqual(map2.get(date1), "date 1 on map 2");
        });
    });

    describe("#setAll", function () {
        it("overrides all map keys", function () {
            var map1,
                map2;

            map1 = new DataMap({
                a: 1,
                b: 2,
                c: 3
            });

            map2 = new DataMap({
                a: 4,
                b: 5,
                c: 6,
                d: 7
            });

            map1.setAll(map2);
            assert.deepEqual(map1.keys(), ["a", "b", "c", "d"]);
            assert.strictEqual(map1.get("a"), 4);
            assert.strictEqual(map1.get("b"), 5);
            assert.strictEqual(map1.get("c"), 6);
            assert.strictEqual(map1.get("d"), 7);
        });
    });

    describe("#values", function () {
        it("retrieves map values", function () {
            var map;

            map = new DataMap({
                a: 1,
                b: 2,
                c: 3,
                d: 2
            });

            assert.deepEqual(map.values().sort(), [1, 2, 2, 3]);
        });
    });
});
