/**
 * @license Copyright (c) 2013, CKSource - Frederico Knabben
 * For licensing, see LICENSE
 */

"use strict";

/* global describe: false, it: false */

var assert = require("assert"),
    Graph = require(global.MODULE_ROOT + "/libraries/scripts/Bender/DataContainer/Graph");

describe("Bender/DataContainer/Repository/Graph", function () {
    describe("#add", function () {
        it("adds graph nodes", function () {
            var graph = new Graph();

            assert.strictEqual(graph.add(1).vertex.value, 1);
            assert.ok(graph.has(1));
            assert.ok(!graph.has(2));
            assert.ok(!graph.has(3));

            assert.strictEqual(graph.add(2).vertex.value, 2);
            assert.ok(graph.has(1));
            assert.ok(graph.has(2));
            assert.ok(!graph.has(3));

            assert.strictEqual(graph.add(3).vertex.value, 3);
            assert.ok(graph.has(1));
            assert.ok(graph.has(2));
            assert.ok(graph.has(3));
        });

        it("does not duplicate nodes", function () {
            var graph = new Graph();

            graph.add(1).next.add(2);
            graph.add(3).next.add(2);

            assert.equal(graph.length, 3);
            assert.ok(graph.has(1));
            assert.ok(graph.has(2));
            assert.ok(graph.has(3));
        });

        it("does not allow duplicate vertices", function () {
            var graph = new Graph();

            graph.add(1);

            assert.throws(function () {
                graph.add(1);
            });
        });

        it("does not allow duplicate nested vertices", function () {
            var graph = new Graph();

            graph.add(1).next.add(2);
            assert.throws(function () {
                graph.get(1).next.add(2);
            });
        });

        it("maintains node values", function () {
            var graph = new Graph();

            graph.add(1).next.add(2);
            graph.add(3).next.add(2);

            assert.strictEqual(graph.get(1).next.get(2).value, 2);
            assert.strictEqual(graph.get(3).next.get(2).value, 2);
        });

        it("maintains cyclic node siblings", function () {
            var graph = new Graph();

            graph.add(1).next.add(2);
            graph.get(2).next.add(3);
            graph.get(3).next.add(1);

            assert.ok(graph.get(1).next.has(2));
            assert.ok(graph.get(1).prev.has(3));

            assert.ok(graph.get(2).next.has(3));
            assert.ok(graph.get(2).prev.has(1));

            assert.ok(graph.get(3).next.has(1));
            assert.ok(graph.get(3).prev.has(2));
        });
    });

    describe("#get", function () {
        it("fails when non-existent vertex is requested", function () {
            var graph = new Graph();

            assert.throws(function () {
                graph.get(0);
            });
        });

        it("tracks graph edges", function () {
            var graph = new Graph();

            graph.add(1).next.add(2);
            graph.add(3).next.add(2);

            assert.equal(graph.length, 3);

            assert.ok(graph.get(1).next.has(2));

            assert.ok(!graph.get(2).next.has(1));
            assert.ok(graph.get(2).prev.has(1));
            assert.ok(!graph.get(2).next.has(3));
            assert.ok(graph.get(2).prev.has(3));

            assert.ok(graph.get(3).next.has(2));
        });
    });

    describe("#has", function () {
        it("allows checking nested nodes", function () {
            var graph = new Graph();

            graph.add(1).next.add(2);

            assert.ok(graph.has(1));
            assert.ok(graph.has(2));
        });
    });

    // describe("#pick", function () {
    //     it("removes filtered node from graph and inserts them into an other graph", function () {
    //         var graph = new Graph(),
    //             other = new Graph();

    //         graph.add(2).next.add(1);
    //         graph.get(2).next.add(3);
    //         graph.add(4).next.add(3);
    //         graph.get(3).next.add(5);

    //         assert.ok(graph.has(1));
    //         assert.ok(graph.get(1).prev.has(2));

    //         assert.ok(graph.has(2));
    //         assert.ok(graph.get(2).next.has(1));
    //         assert.ok(graph.get(2).next.has(3));

    //         assert.ok(graph.has(3));
    //         assert.ok(graph.get(3).next.has(5));
    //         assert.ok(graph.get(3).prev.has(2));
    //         assert.ok(graph.get(3).prev.has(4));

    //         assert.ok(graph.has(4));
    //         assert.ok(graph.get(4).next.has(3));

    //         assert.ok(graph.has(5));
    //         assert.ok(graph.get(5).prev.has(3));

    //         graph.pick(other, function (nodeValue, node) {
    //             return nodeValue === 3;
    //         });

    //         assert.ok(other.has(3));
    //         assert.ok(!other.get(3).prev.has(4));
    //         assert.ok(!other.get(3).next.has(5));

    //         assert.ok(graph.has(1));
    //         assert.ok(graph.get(1).prev.has(2));

    //         assert.ok(graph.has(2));
    //         assert.ok(graph.get(2).next.has(1));
    //         assert.ok(!graph.get(2).next.has(3));

    //         assert.ok(!graph.has(3));

    //         assert.ok(graph.has(4));
    //         assert.ok(!graph.get(4).next.has(3));

    //         assert.ok(graph.has(5));
    //         assert.ok(!graph.get(5).prev.has(3));
    //     });
    // });

    describe("#put", function () {
        it("inserts node if it is not already inserted", function () {
            var encapsulatedObject = {},
                graph = new Graph();

            assert.strictEqual(graph.add(encapsulatedObject).vertex.value, encapsulatedObject);
        });

        it("uses node if it is already inserted", function () {
            var encapsulatedObject = {},
                graph = new Graph();

            graph.add(encapsulatedObject);
            assert.strictEqual(graph.put(encapsulatedObject).vertex.value, encapsulatedObject);
        });
    });

    describe("#putAll", function () {
        it("sets all nodes from other graph", function () {
            var graphA = new Graph(),
                graphB = new Graph();

            graphA.add(2).next.add(1);
            graphA.add(4);
            graphA.add(5);

            graphB.add(2).next.add(3);
            graphB.add(3).next.add(5);
            graphB.add(4).next.add(3);

            graphA.putAll(graphB);

            assert.ok(graphA.length, 5);
            assert.ok(graphA.has(1));
            assert.ok(graphA.has(2));
            assert.ok(graphA.get(2).next.has(1));
            assert.ok(graphA.get(2).next.has(3));
            assert.ok(graphA.has(3));
            assert.ok(graphA.get(3).next.has(5));
            assert.ok(graphA.has(4));
            assert.ok(graphA.get(4).next.has(3));
            assert.ok(graphA.has(5));
        });
    });

    describe("#remove", function () {
        it("removes nested vertice", function () {
            var graph = new Graph();

            graph.add(1).next.add(2);

            assert.ok(graph.has(1));
            assert.ok(graph.has(2));

            graph.remove(2);

            assert.ok(graph.has(1));
            assert.ok(!graph.get(1).next.has(2));

            assert.ok(!graph.has(2));
        });

        it("removes node and its connections", function () {
            var graph = new Graph();

            graph.add(2).next.add(1);
            graph.get(2).next.add(3);
            graph.add(4).next.add(3);
            graph.get(3).next.add(5);

            assert.ok(graph.has(1));
            assert.ok(graph.get(1).prev.has(2));

            assert.ok(graph.has(2));
            assert.ok(graph.get(2).next.has(1));
            assert.ok(graph.get(2).next.has(3));

            assert.ok(graph.has(3));
            assert.ok(graph.get(3).next.has(5));
            assert.ok(graph.get(3).prev.has(2));
            assert.ok(graph.get(3).prev.has(4));

            assert.ok(graph.has(4));
            assert.ok(graph.get(4).next.has(3));

            assert.ok(graph.has(5));
            assert.ok(graph.get(5).prev.has(3));

            graph.remove(2);
            graph.remove(4);

            assert.ok(graph.has(1));
            assert.ok(!graph.get(1).prev.has(2));

            assert.ok(!graph.has(2));

            assert.ok(graph.has(3));
            assert.ok(!graph.get(3).prev.has(2));
            assert.ok(!graph.get(3).prev.has(4));

            assert.ok(!graph.has(4));

            assert.ok(graph.has(5));
            assert.ok(graph.get(5).prev.has(3));
        });
    });
});
