/**
 * @license Copyright (c) 2013, Cheng Lou (github.com/chenglou)
 * @license Copyright (c) 2013, CKSource - Frederico Knabben
 * For licensing, see LICENSE
 */

"use strict";

/* global define: false */

(function (factory) {

    if ("function" === typeof define) {
        define([
            "configuration-processor"
        ], factory);
    } else {
        module.exports = factory.apply(null, [
            require("configuration-processor")
        ]);
    }

}(function (configurationProcessor) {

    var config = configurationProcessor.configurationProcessor,
        Trie, // constructor, function
        WORD_END = "end";

    /**
     * Good for fast insertion/removal/lookup of strings.
     *
     * @constructor
     * @example
     *  var trie = new Trie(['bear', 'beer']);
     *  trie.add('hello'); // => 'hello'
     *  trie.add('helloha!'); // => 'helloha!'
     *  trie.has('bears'); // => false
     *  trie.longestPrefixOf('beatrice'); // => 'bea'
     *  trie.wordsWithPrefix('hel'); // => ['hello', 'helloha!']
     *  trie.remove('beers'); // => undefined. 'beer' still exists
     *  trie.remove('Beer') // => undefined. Case-sensitive
     *  trie.remove('beer') // => 'beer'. Removed
     * @param {array} [words]
     */
    Trie = function (words) {
        this.root = {};
        this.size = 0;

        if (words) {
            config.assertArray(words).forEach(this.add, this);
        }
    };

    /**
     * Add a whole string to the trie.
     *
     * @param {string} word
     * @return {string} added word
     */
    Trie.prototype.add = function (word) {
        var currentNode,
            i,
            letter;

        word = config.assertString(word);

        this.size += 1;

        currentNode = this.root;
        for (i = 0; i < word.length; i += 1) {
            letter = word[i];
            if (!currentNode[letter]) {
                currentNode[letter] = {};
            }
            currentNode = currentNode[letter];
        }
        currentNode[WORD_END] = true;

        return word;
    };

    /**
     * @param {string} word
     * @return {bool}
     */
    Trie.prototype.has = function (word) {
        var currentNode,
            i,
            letter;

        word = config.assertString(word);

        currentNode = this.root;
        for (i = 0; i < word.length; i += 1) {
            letter = word[i];
            if (!currentNode[letter]) {
                return false;
            }
            currentNode = currentNode[letter];
        }

        return !!currentNode[WORD_END];
    };

    /**
     * Find all words containing the prefix. The word itself counts as a prefix.
     *
     * @example
     *  var trie = new Trie;
     *  trie.add('hello');
     *  trie.longestPrefixOf('he'); // 'he'
     *  trie.longestPrefixOf('hello'); // 'hello'
     *  trie.longestPrefixOf('helloha!'); // 'hello'
     * @return {string}
     */
    Trie.prototype.longestPrefixOf = function (word) {
        var currentNode,
            i,
            letter,
            prefix;

        word = config.assertString(word);

        currentNode = this.root;
        prefix = '';

        for (i = 0; i < word.length; i += 1) {
            letter = word[i];
            if (!currentNode[letter]) {
                break;
            }
            prefix += letter;
            currentNode = currentNode[letter];
        }

        return prefix;
    };

    /**
     * Find all words containing the prefix. The word itself counts as a prefix.
     *
     * @example
     *  var trie = new Trie;
     *  trie.wordsWithPrefix(''); // []. Check later case below.
     *  trie.add('');
     *  trie.wordsWithPrefix(''); // ['']
     *  trie.add('he');
     *  trie.add('hello');
     *  trie.add('hell');
     *  trie.add('bear');
     *  trie.add('z');
     *  trie.add('zebra');
     *  trie.wordsWithPrefix('hel'); // ['hell', 'hello']
     * @return {array} an array of strings, or empty array if no word found.
     */
    Trie.prototype.wordsWithPrefix = function (prefix) {
        var accumulatedLetters,
            currentNode,
            i,
            letter,
            node,
            queue,
            queueHead,
            subNode,
            words;

        prefix = config.assertString(prefix);

        words = [];
        currentNode = this.root;
        for (i = 0; i < prefix.length; i += 1) {
            letter = prefix[i];
            currentNode = currentNode[letter];
            if (!currentNode) {
                return [];
            }
        }

        queue = [];
        queue.push([currentNode, '']);
        while (queue.length > 0) {
            queueHead = queue.shift();
            node = queueHead[0];
            accumulatedLetters = queueHead[1];

            if (node[WORD_END]) {
                words.push(prefix + accumulatedLetters);
            }
            for (letter in node) {
                if (node.hasOwnProperty(letter)) {
                    subNode = node[letter];
                    queue.push([subNode, accumulatedLetters + letter]);
                }
            }
        }

        return words;
    };

    /*
     * @return {string}
     *  the string removed, or undefined if the word in its whole doesn't
     *  exist. **Note:** this means removing `beers` when only `beer` exists will
     *  return undefined and conserve `beer`.
     */
    Trie.prototype.remove = function (word) {
        var currentNode,
            i,
            letter,
            prefix;

        word = config.assertString(word);

        currentNode = this.root;
        prefix = [];

        for (i = 0; i < word.length; i += 1) {
            letter = word[i];
            if (!currentNode[letter]) {
                return;
            }
            currentNode = currentNode[letter];
            prefix.push([letter, currentNode]);
        }
        if (!currentNode[WORD_END]) {
            return;
        }

        this.size--;
        delete currentNode[WORD_END];

        return word;
    };

    return Trie;

}));
