/**
 * @license Copyright (c) 2013, CKSource - Frederico Knabben
 * For licensing, see LICENSE
 */

"use strict";

/* global define: false */

(function (factory) {

    if ("function" === typeof define) {
        define([
            "configuration-processor/configuration-processor",
            "data-container/Node/LinkedList",
            "data-container/Repository"
        ], factory);
    } else {
        module.exports = factory.apply(null, [
            require("configuration-processor"),
            require(__dirname + "/../Node/LinkedList"),
            require(__dirname + "/../Repository")
        ]);
    }

}(function (configurationProcessor, LinkedListNode, Repository) {

    var assertComparator, // private, function
        genericComparator, // private, function
        config = configurationProcessor.configurationProcessor,
        repairListNodePointers, // private, function
        LinkedList; // constructor, function

    /**
     * @access private
     * @param {mixed} itemA
     * @param {mixed} itemB
     * @return {number}
     */
    genericComparator = function (itemA, itemB) {
        if (itemA < itemB) {
            return -1;
        }

        if (itemA > itemB) {
            return 1;
        }

        return 0;
    };

    /**
     * @access private
     * @param {function} [comparator]
     * @return {function}
     * @throws {Error} if input argument is invalid
     */
    assertComparator = config.expectDefault(config.expectFunction(), genericComparator);

    /**
     * @access private
     * @param {Bender/DataContainer/Repository/LinkedList} linkedList
     * @return {void}
     */
    repairListNodePointers = function (linkedList) {
        var i,
            list = linkedList.list,
            listNode;

        if (list.length < 1) {
            return;
        }

        listNode = list[0];
        for (i = 1; i < list.length; i += 1) {
            listNode = listNode.next = list[i];
        }
    };

    /**
     * @auguments Bender/DataContainer/Repository
     * @constructor
     * @param {array} [initialList]
     */
    LinkedList = function (initialList) {
        Repository.call(this);

        if (initialList) {
            config.assertArray(initialList).forEach(this.add, this);
        }
    };
    LinkedList.prototype = Object.create(Repository.prototype);

    /**
     * @param {mixed} item
     * @return {Bender/DataContainer/LinkedList}
     */
    LinkedList.prototype.add = function (item) {
        item = this.encapsulateItem(item);
        if (!this.isEmpty()) {
            item.previous = this.last;
            this.last.next = item;
        }

        return Repository.prototype.add.call(this, item);
    };

    /**
     * @param {mixed} item
     * @return {bool}
     */
    LinkedList.prototype.canDecapsulate = function (item) {
        return item instanceof LinkedListNode;
    };

    /**
     * @param {Bender/DataContainer/LinkedList} item
     * @return {mixed}
     * @throws {Error} if item cannot be decapsulated
     */
    LinkedList.prototype.decapsulateItem = function (item) {
        if (!this.canDecapsulate(item)) {
            throw new Error("Item cannot be decapsulated: " + item);
        }

        return item.value;
    };

    /**
     * @param {mixed} item
     * @return {Bender/DataContainer/LinkedList}
     */
    LinkedList.prototype.encapsulateItem = function (item) {
        return new LinkedListNode(item);
    };

    /**
     * @param {mixed} item
     * @return {Bender/DataContainer/LinkedList}
     */
    LinkedList.prototype.indexOf = function (item) {
        var i,
            list = this.list;

        for (i = 0; i < list.length; i += 1) {
            if (list[i].value === item) {
                return i;
            }
        }

        return -1;
    };

    /**
     * @param {Bender/DataContainer/Repository/LinkedListNode} linkedListNode
     * @return {bool}
     */
    LinkedList.prototype.isItemAllowed = function (linkedListNode) {
        return linkedListNode instanceof LinkedListNode;
    };

    /**
     * @param {mixed} item
     * @return {mixed} removed item
     * @throws {Error} if this repository does not hold given item
     */
    LinkedList.prototype.remove = function (item) {
        var node = this.at(this.indexOf(item));

        if (node.previous) {
            node.previous.next = node.next;
        }
        if (node.next) {
            node.next.previous = node.previous;
        }

        return Repository.prototype.remove.call(this, item);
    };

    /**
     * @param {comparator}
     * @return {void}
     */
    LinkedList.prototype.sort = function (comparator) {
        this.list.sort(assertComparator(comparator));
        repairListNodePointers(this);
    };

    /**
     * @param {mixed} item
     * @return {Bender/DataContainer/LinkedList}
     */
    LinkedList.prototype.unshift = function (item) {
        item = this.encapsulateItem(item);

        if (!this.isEmpty()) {
            this.head.previous = item;
            item.next = this.head;
        }

        return Repository.prototype.unshift.call(this, item);
    };

    return LinkedList;

}));
