/**
 * @license Copyright (c) 2013, CKSource - Frederico Knabben
 * For licensing, see LICENSE
 */

"use strict";

/* global define: false */

(function (factory) {

    if ("function" === typeof define) {
        define([
            "configuration-processor/configuration-processor"
        ], factory);
    } else {
        module.exports = factory.apply(null, [
            require("configuration-processor")
        ]);
    }

}(function (configurationProcessor) {

    var config = configurationProcessor.configurationProcessor,
        Repository, // constructor, function
        assertArray; // private, function

    /**
     * @access private
     * @param {mixed} item
     * @return {mixed}
     * @throws {Error} if item is not an array
     */
    assertArray = config.expectArray();

    /**
     * @constructor
     * @param {array} items
     */
    Repository = function (items) {
        this.list = [];

        if (items) {
            items = assertArray(items);
            items.forEach(this.add, this);
        }
    };

    Object.defineProperties(Repository.prototype, {
        head: {
            /**
             * @return {mixed}
             */
            get: function () {
                return this.list[0];
            }
        },
        last: {
            /**
             * @return {mixed}
             */
            get: function () {
                return this.list[this.list.length - 1];
            }
        },
        length: {
            /**
             * @return {number}
             */
            get: function () {
                return this.list.length;
            }
        }
    });

    /**
     * @param {mixed} item
     * @return {mixed} added item
     * @throws {Error} if item cannot be added to container
     */
    Repository.prototype.add = function (item) {
        if (!this.canAdd(item)) {
            throw new Error("Item cannot be added to container: " + item);
        }

        this.list.push(item);

        return item;
    };

    /**
     * @param {number} index
     * @return {mixed}
     * @throws {Error} if index is out of range
     */
    Repository.prototype.at = function (index) {
        if (!this.hasIndex(index)) {
            throw new Error("Index is out of container range: " + index);
        }

        return this.list[index];
    };

    /**
     * @param {mixed} item
     * @return {bool}
     */
    Repository.prototype.canAdd = function (item) {
        return this.isItemAllowed(item) && !this.has(item);
    };

    /**
     * @param {mixed} item
     * @return {bool}
     */
    Repository.prototype.canGet = function (item) {
        return this.has(item);
    };

    /**
     * @param {mixed} item
     * @return {bool}
     */
    Repository.prototype.canRemove = function (item) {
        return this.has(item);
    };

    /**
     * This method is almost useless when Repository object is not augumented. It
     * allows some flexibility for child objects.
     * It can also be used to instantlt throw exception when expected item is not
     * placed inside repository.
     *
     * @param {mixed} item
     * @return {bool}
     * @see Bender/DataContainer/Repository/LinkedList
     * @throws {Error} if item cannot be fetched from repository
     */
    Repository.prototype.get = function (item) {
        if (!this.canGet(item)) {
            throw new Error("Item cannot be fetched from repository: " + item);
        }

        return this.at(this.indexOf(item));
    };

    /**
     * @param {mixed} item
     * @return {bool}
     */
    Repository.prototype.has = function (item) {
        return -1 !== this.indexOf(item);
    };

    /**
     * @param {int} index
     * @return {bool}
     */
    Repository.prototype.hasIndex = function (index) {
        index = config.assertInt(index);

        return 0 <= index && index < this.list.length;
    };

    /**
     * @param {mixed} item
     * @return {int} index of item on the list (same as indexOf)
     */
    Repository.prototype.indexOf = function (item) {
        return this.list.indexOf(item);
    };

    /**
     * @return {bool}
     */
    Repository.prototype.isEmpty = function () {
        return this.list.length < 1;
    };

    /**
     * @param {mixed} item
     * @return {bool}
     */
    Repository.prototype.isItemAllowed = function (item) {
        return true;
    };

    /**
     * @param {mixed} item
     * @return {mixed} removed item
     * @throws {Error} if this repository does not hold given item
     */
    Repository.prototype.remove = function (item) {
        if (!this.canRemove(item)) {
            throw new Error("Item cannot be removed from repository: " + item);
        }

        this.list.splice(this.indexOf(item), 1);

        return item;
    };

    /**
     * @param {mixed} item
     * @return {mixed} shifted item
     */
    Repository.prototype.shift = function () {
        return this.remove(this.head);
    };

    /**
     * @param {mixed} item
     * @return {mixed} unshifted item
     * @throws {Error} if item cannot be added to container
     */
    Repository.prototype.unshift = function (item) {
        if (!this.canAdd(item)) {
            throw new Error("Item cannot be added to container: " + item);
        }

        this.list.unshift(item);

        return item;
    };

    return Repository;

}));
