/**
 * @license Copyright (c) 2013, CKSource - Frederico Knabben
 * For licensing, see LICENSE
 */

"use strict";

/* global define: false */

(function (factory) {

    if ("function" === typeof define) {
        define([
            "data-container/GraphVertex",
            "data-container/Map/GraphVertex",
            "data-container/Node"
        ], factory);
    } else {
        module.exports = factory.apply(null, [
            require(__dirname + "/../GraphVertex"),
            require(__dirname + "/../Map/GraphVertex"),
            require(__dirname + "/../Node")
        ]);
    }

}(function (GraphVertex, GraphVertexMap, Node) {

    var GraphNode; // constructor, function

    /**
     * @auguments Bender/DataContainer/Node
     * @constructor
     * @param {Bender/DataContainer/GraphVertex} graphVertex
     * @param {Bender/DataContainer/Map/GraphVertex} predecessors
     * @param {Bender/DataContainer/Map/GraphVertex} successors
     */
    GraphNode = function (graphVertex, predecessors, successors) {
        Node.call(this, graphVertex);

        this.next = successors;
        this.prev = predecessors;

        Object.defineProperties(this, {
            vertex: {
                /**
                 * @return {Bender/DataContainer/GraphVertex}
                 */
                get: function () {
                    return graphVertex;
                }
            }
        });
    };
    GraphNode.prototype = Object.create(Node.prototype);

    /**
     * Removes node both from vertice predecessors and successors.
     *
     * @param {mixed} nodeValue
     * @return {void}
     * @throws {Error} if node does not contain sibling vertice
     */
    GraphNode.prototype.removeSibling = function (nodeValue) {
        var has;

        if (this.next.has(nodeValue)) {
            has = true;
            this.next.remove(nodeValue);
        }
        if (this.prev.has(nodeValue)) {
            has = true;
            this.prev.remove(nodeValue);
        }

        if (!has) {
            throw new Error("Node does not contain sibling vertice: " + nodeValue);
        }
    };

    /**
     * @param {mixed} nodeValue
     * @return {bool}
     */
    GraphNode.prototype.hasSibling = function (nodeValue) {
        return this.next.has(nodeValue) || this.prev.has(nodeValue);
    };

    /**
     * @param {mixed} item
     * @return {bool}
     */
    GraphNode.prototype.isItemAllowed = function (item) {
        return item instanceof GraphVertex;
    };

    /**
     * @param {Bender/DataContainer/Map/GraphVertex|void} graphVertexMap
     * @return {bool}
     */
    GraphNode.prototype.isSiblingAllowed = function (graphVertexMap) {
        return graphVertexMap instanceof GraphVertexMap;
    };

    return GraphNode;

}));
