/**
 * @license Copyright (c) 2013, CKSource - Frederico Knabben
 * For licensing, see LICENSE
 */

"use strict";

/* global define: false */

(function (factory) {

    if ("function" === typeof define) {
        define([
            "configuration-processor/configuration-processor",
            "data-container/Map",
            "data-container/GraphVertex"
        ], factory);
    } else {
        module.exports = factory.apply(null, [
            require("configuration-processor"),
            require(__dirname + "/../Map"),
            require(__dirname + "/../GraphVertex")
        ]);
    }

}(function (configurationProcessor, DataMap, GraphVertex) {

    var config = configurationProcessor.configurationProcessor,
        GraphVertexMap; // constructor, function

    /**
     * @auguments Bender/DataContainer/Map
     * @constructor
     * @param {Bender/DataContainer/Map/GraphVertex} verticesGroup
     */
    GraphVertexMap = function (verticesGroup) {
        DataMap.call(this);

        if (verticesGroup) {
            this.verticesGroup = config.assertInstanceOf(GraphVertexMap, verticesGroup);
        }
    };
    GraphVertexMap.prototype = Object.create(DataMap.prototype);

    /**
     * @param {mixed} nodeValue
     * @return {Bender/DataContainer/GraphVertex}
     * @throws {Error} if graph node map already contains vertice
     */
    GraphVertexMap.prototype.add = function (nodeValue) {
        if (this.has(nodeValue)) {
            throw new Error("Graph node map already contains vertice: " + nodeValue);
        }

        return this.set(nodeValue, this.encapsulateItem(nodeValue));
    };

    /**
     * @param {mixed} nodeValue
     * @return {Bender/DataContainer/Vertex/Graph}
     */
    GraphVertexMap.prototype.encapsulateItem = function (nodeValue) {
        return new GraphVertex(nodeValue);
    };

    /**
     * @param {Bender/DataContainer/GraphVertex} graphVertex
     * @return {bool}
     */
    GraphVertexMap.prototype.isItemAllowed = function (graphVertex) {
        return graphVertex instanceof GraphVertex;
    };

    /**
     * @param {mixed} nodeValue
     * @return {Bender/DataContainer/GraphVertex}
     */
    GraphVertexMap.prototype.put = function (nodeValue) {
        if (this.has(nodeValue)) {
            return this.get(nodeValue);
        }

        return this.add(nodeValue);
    };

    /**
     * @param {Bender/DataContainer/Map/GraphVertex} other
     * @return {void}
     */
    GraphVertexMap.prototype.putAll = function (other) {
        other = config.assertInstanceOf(GraphVertexMap, other);
        other.forEach(function (nodeValue, node) {
            this.put(nodeValue);
        }, this);
    };

    /**
     * @param {mixed} key
     * @param {Bender/DataContainer/GraphVertex} vertex
     * @return {Bender/DataContainer/GraphVertex}
     */
    GraphVertexMap.prototype.set = function (key, vertex) {
        vertex = DataMap.prototype.set.call(this, key, vertex);
        if (this.verticesGroup) {
            this.verticesGroup.set(key, vertex);
        }

        return vertex;
    };

    return GraphVertexMap;

}));
