/**
 * @license Copyright (c) 2013, CKSource - Frederico Knabben
 * For licensing, see LICENSE
 */

"use strict";

/* global define: false */

(function (factory) {

    if ("function" === typeof define) {
        define([
            "configuration-processor/configuration-processor",
            "data-container/Map",
            "data-container/Node/Graph",
            "data-container/Map/GraphVertex"
        ], factory);
    } else {
        module.exports = factory.apply(null, [
            require("configuration-processor"),
            require(__dirname + "/../Map"),
            require(__dirname + "/../Node/Graph"),
            require(__dirname + "/GraphVertex")
        ]);
    }

}(function (configurationProcessor, DataMap, GraphNode, GraphVertexMap) {

    var config = configurationProcessor.configurationProcessor,
        GraphNodeMap; // constructor, function

    /**
     * @auguments Bender/DataContainer/Map
     * @constructor
     * @param {Bender/DataContainer/Map/GraphVertex} vertices
     */
    GraphNodeMap = function (vertices) {
        DataMap.call(this);

        this.vertices = config.assertInstanceOf(GraphVertexMap, vertices);
    };
    GraphNodeMap.prototype = Object.create(DataMap.prototype);

    /**
     * @param {mixed} nodeValue
     * @return {Bender/DataContainer/GraphVertex}
     * @throws {Error} if graph node map already contains vertice
     */
    GraphNodeMap.prototype.add = function (nodeValue) {
        if (this.has(nodeValue)) {
            throw new Error("Graph node map already contains vertice: " + nodeValue);
        }

        return this.set(nodeValue, this.encapsulateItem(nodeValue));
    };

    /**
     * @param {mixed} nodeValue
     * @return {Bender/DataContainer/Vertex/Graph}
     */
    GraphNodeMap.prototype.encapsulateItem = function (nodeValue) {
        var vertex = this.vertices.put(nodeValue),
            predecessors = this.encapsulateVertices(this.vertices),
            successors = this.encapsulateVertices(this.vertices);

        return new GraphNode(vertex, predecessors, successors);
    };

    /**
     * @param {Bender/DataContainer/Map/GraphVertex} parentVertices
     * @return {Bender/DataContainer/Map/GraphVertex}
     */
    GraphNodeMap.prototype.encapsulateVertices = function (parentVertices) {
        parentVertices = config.assertInstanceOf(GraphVertexMap, parentVertices);

        return new GraphVertexMap(parentVertices);
    };

    /**
     * @param {Bender/DataContainer/Node/Graph} graphNode
     * @return {bool}
     */
    GraphNodeMap.prototype.isItemAllowed = function (graphNode) {
        return graphNode instanceof GraphNode;
    };

    /**
     * @param {mixed} nodeValue
     * @return {Bender/DataContainer/GraphVertex}
     */
    GraphNodeMap.prototype.put = function (nodeValue) {
        if (this.has(nodeValue)) {
            return this.get(nodeValue);
        }

        return this.add(nodeValue);
    };

    return GraphNodeMap;

}));
