/**
 * @license Copyright (c) 2013, Cheng Lou (github.com/chenglou)
 * @license Copyright (c) 2013, CKSource - Frederico Knabben
 * For licensing, see LICENSE
 */

"use strict";

/* global define: false */

(function (factory) {

    if ("function" === typeof define) {
        define([
            "configuration-processor/configuration-processor"
        ], factory);
    } else {
        module.exports = factory.apply(null, [
            require("configuration-processor")
        ]);
    }

}(function (configurationProcessor) {

    var config = configurationProcessor.configurationProcessor,
        REGEXP_OBJECT_TYPE = (/\[object ([^\]]+)\]/),
        DataMap,
        assertItemIsAllowed, // private, function
        assertKeyIsAllowed, // private, function
        extractDataType, // private, function
        has, // private, function
        hash, // private, function
        hashSimple, // private, function
        hashSpecial, // private, function
        simpleHashableTypes = {
            "Boolean": true,
            "Function": true,
            "Null": true,
            "Number": true,
            "RegExp": true,
            "String": true,
            "Undefined": true
        };

    /**
     * @access private
     * @param {Bender/DataContainer/Map} map
     * @param {mixed} key
     * @return {mixed} input key
     * @throws {Error} if given key is not allowed
     */
    assertKeyIsAllowed = function (map, key) {
        if (map.isKeyAllowed(key)) {
            return key;
        }

        throw new Error("Map does not allow given key: " + key);
    };

    /**
     * @access private
     * @param {Bender/DataContainer/Map} map
     * @param {mixed} value
     * @return {mixed} input value
     * @throws {Error} if given value is not allowed
     */
    assertItemIsAllowed = function (map, value) {
        if (map.isItemAllowed(value)) {
            return value;
        }

        throw new Error("Map does not allow given value: " + value);
    };

    /**
     * @access private
     * @param {Bender/DataContainer/Map} map
     * @param {mixed} type
     * @return {string}
     */
    extractDataType = function (map, type) {
        return Object.prototype.toString.apply(type).match(REGEXP_OBJECT_TYPE)[1];
    };

    /**
     * @access private
     * @param {Bender/DataContainer/Map} map
     * @param {mixed} key
     * @param {string} hashedKey
     * @return {string}
     */
    has = function (map, key, hashedKey) {
        if (!hashedKey) {
            hashedKey = hash(map, key);
        }

        return map.hashes.hasOwnProperty(hashedKey);
    };

    /**
     * @access private
     * @param {Bender/DataContainer/Map} map
     * @param {mixed} key
     * @return {string}
     */
    hash = function (map, key) {
        var type = extractDataType(map, key);

        if (simpleHashableTypes[extractDataType(map, key)]) {
            return hashSimple(map, key, type);
        }

        return hashSpecial(map, key, type);
    };

    /**
     * @access private
     * @param {Bender/DataContainer/Map} map
     * @param {mixed} key
     * @param {string} type
     * @return {string}
     */
    hashSimple = function (map, key, type) {
        var typedKey = type + key;

        if (map.simpleTypeHashes.hasOwnProperty(typedKey)) {
            return map.simpleTypeHashes[typedKey];
        }

        map.hashIdTracker += 1;
        map.simpleTypeHashes[typedKey] = map.hashIdTracker;

        return map.hashIdTracker;
    };

    /**
     * @access private
     * @param {Bender/DataContainer/Map} map
     * @param {mixed} key
     * @param {string} type
     * @return {string}
     */
    hashSpecial = function (map, key, type) {
        var i;

        for (i = 0; i < map.specialTypeHashes.length; i += 1) {
            if (Object.is(map.specialTypeHashes[i].key, key)) {
                return map.specialTypeHashes[i].hash;
            }
        }

        map.hashIdTracker += 1;
        map.specialTypeHashes.push({
            hash: map.hashIdTracker,
            key: key
        });

        return map.hashIdTracker;
    };

    /**
     * Kind of a stopgap measure for the upcoming JavaScript Map
     *
     * @constructor
     * @param {object} [objectToMap] initial object to hash
     */
    DataMap = function (objectToMap) {
        var key;

        this.hashes = {};
        this.hashIdTracker = 0;
        this.simpleTypeHashes = {};
        this.specialTypeHashes = [];

        if (objectToMap) {
            objectToMap = config.assertObject(objectToMap);
            for (key in objectToMap) {
                if (objectToMap.hasOwnProperty(key)) {
                    this.set(key, objectToMap[key]);
                }
            }
        }
    };

    Object.defineProperty(DataMap.prototype, "length", {
        /**
         * @return {number}
         */
        get: function () {
            return Object.keys(this.hashes).length;
        }
    });

    /**
     * @param {Bender/DataContainer/Map} other map that is going to absorb keys
     * @param {function} callback
     * @param {object} [thisArg]
     * @return {Bender/DataContainer/Map}
     */
    DataMap.prototype.filter = function (other, callback, thisArg) {
        callback = config.assertFunction(callback).bind(thisArg);
        other = config.assertInstanceOf(DataMap, other);

        this.forEach(function (key, value) {
            if (true === callback(key, value)) {
                other.set(key, value);
            }
        });
    };

    /**
     * @param {function} callback
     * @param {object} [thisArg]
     * @return {void}
     */
    DataMap.prototype.forEach = function (callback, thisArg) {
        var hash,
            storedItem,
            that = this;

        callback = config.assertFunction(callback);

        for (hash in this.hashes) {
            if (this.hashes.hasOwnProperty(hash)) {
                storedItem = that.hashes[hash];

                callback.call(thisArg, storedItem.key, storedItem.value);
            }
        }
    };

    /**
     * @param {mixed} key
     * @return {mixed}
     */
    DataMap.prototype.get = function (key) {
        var hashedKey = hash(this, assertKeyIsAllowed(this, key));

        if (!has(this, key, hashedKey)) {
            throw new Error("Map does not contain key: " + hashedKey);
        }

        return this.hashes[hashedKey].value;
    };

    /**
     * Check whether a value exists for the key.
     *
     * @param {mixed} key
     * @return {bool}
     */
    DataMap.prototype.has = function (key) {
        return this.isKeyAllowed(key) && has(this, key);
    };

    /**
     * @param {mixed} item
     * @return {bool}
     */
    DataMap.prototype.isItemAllowed = function (item) {
        return true;
    };

    /**
     * @param {mixed} item
     * @return {bool}
     */
    DataMap.prototype.isKeyAllowed = function (key) {
        return true;
    };

    /**
     * @return {array}
     */
    DataMap.prototype.keys = function () {
        var property,
            ret = [];

        for (property in this.hashes) {
            if (this.hashes.hasOwnProperty(property)) {
                ret.push(this.hashes[property].key);
            }
        }

        return ret;
    };

    /**
     * @param {Bender/DataContainer/Map} other map that is going to absorb keys
     * @param {function} callback
     * @param {object} [thisArg]
     * @return {Bender/DataContainer/Map}
     */
    DataMap.prototype.pick = function (other, callback, thisArg) {
        var that = this;

        callback = config.assertFunction(callback).bind(thisArg);
        other = config.assertInstanceOf(DataMap, other);

        this.filter(other, function (key, value) {
            var ret = callback(key, value);

            if (true === ret) {
                that.remove(key);
            }

            return ret;
        });
    };

    /**
     * @param {mixed} key
     * @return {mixed} removed item
     */
    DataMap.prototype.remove = function (key) {
        var hashedKey = hash(this, assertKeyIsAllowed(this, key)),
            removedItem;

        if (!has(this, key, hashedKey)) {
            throw new Error("Map does not contain key: " + hashedKey);
        }

        removedItem = this.hashes[hashedKey].value;
        delete this.hashes[hashedKey];

        return removedItem;
    };

    /**
     * @param {mixed} key
     * @param {mixed} value
     * @return {void}
     */
    DataMap.prototype.set = function (key, value) {
        var hashedKey = hash(this, assertKeyIsAllowed(this, key));

        value = assertItemIsAllowed(this, value);

        this.hashes[hashedKey] = {
            key: key,
            value: value
        };

        return value;
    };

    /**
     * @param {Bender/DataContainer/Map} other
     * @return {void}
     */
    DataMap.prototype.setAll = function (other) {
        other = config.assertInstanceOf(DataMap, other);
        other.forEach(this.set, this);
    };

    /**
     * @return {array}
     */
    DataMap.prototype.values = function () {
        var property,
            ret = [];

        for (property in this.hashes) {
            if (this.hashes.hasOwnProperty(property)) {
                ret.push(this.hashes[property].value);
            }
        }

        return ret;
    };

    return DataMap;

}));
