/**
 * @license Copyright (c) 2013, CKSource - Frederico Knabben
 * For licensing, see LICENSE
 */

"use strict";

/* global define: false */

(function (factory) {

    if ("function" === typeof define) {
        define([
            "configuration-processor/configuration-processor",
            "data-container/Map/GraphNode",
            "data-container/Map/GraphVertex"
        ], factory);
    } else {
        module.exports = factory.apply(null, [
            require("configuration-processor"),
            require(__dirname + "/Map/GraphNode"),
            require(__dirname + "/Map/GraphVertex")
        ]);
    }

}(function (configurationProcessor, GraphNodeMap, GraphVertexMap) {

    var config = configurationProcessor.configurationProcessor,
        Graph, // constructor, function
        findPredecessors; // private, function

    /**
     * @access private
     * @param {Bender/DataContainer/Graph} graph
     * @param {mixed} nodeValue
     * @return {Bender/DataContainer/Map/GraphVertex}
     */
    findPredecessors = function (graph, nodeValue) {
        var predecessors;

        predecessors = new GraphVertexMap(graph.vertices);
        graph.nodes.forEach(function (index, node) {
            if (node.next.has(nodeValue)) {
                predecessors.set(index, node.value);
            }
        });

        return predecessors;
    };

    /**
     * @constructor
     */
    Graph = function () {
        this.vertices = this.encapsulateVertexMap();
        this.nodes = this.encapsulateNodeMap(this.vertices);
    };

    Object.defineProperty(Graph.prototype, "length", {
        /**
         * @return {number}
         */
        get: function () {
            return this.vertices.length;
        }
    });

    /**
     * @param {mixed} nodeValue
     * @return {Bender/DataContainer/Node/Graph}
     */
    Graph.prototype.add = function (nodeValue) {
        return this.nodes.add(nodeValue);
    };

    /**
     * @param {mixed} [nodeMap]
     * @return {Bender/DataContainer/Map/GraphNode}
     */
    Graph.prototype.encapsulateNodeMap = function (nodeMap) {
        return new GraphNodeMap(nodeMap);
    };

    /**
     * @return {Bender/DataContainer/Map/GraphVertex}
     */
    Graph.prototype.encapsulateVertexMap = function () {
        return new GraphVertexMap();
    };

    /**
     * @param {mixed} nodeValue
     * @return {Bender/DataContainer/Node/Graph}
     * @throws {Error} if graph does not contain vertex
     */
    Graph.prototype.get = function (nodeValue) {
        var node;

        if (this.nodes.has(nodeValue)) {
            node = this.nodes.get(nodeValue);
        } else if (this.vertices.has(nodeValue)) {
            node = this.nodes.add(nodeValue);
        } else {
            throw new Error("Graph does not contain vertex: " + nodeValue);
        }

        node.prev = findPredecessors(this, nodeValue);

        return node;
    };

    /**
     * @param {mixed} nodeValue
     * @return {bool}
     */
    Graph.prototype.has = function (nodeValue) {
        return this.nodes.has(nodeValue) || this.vertices.has(nodeValue);
    };

    /**
     * @param {mixed} nodeValue
     * @return {Bender/DataContainer/Node/Graph}
     */
    Graph.prototype.put = function (nodeValue) {
        if (this.has(nodeValue)) {
            return this.get(nodeValue);
        }

        return this.add(nodeValue);
    };

    /**
     * @param {Bender/DataContainer/Graph} other
     * @return {Bender/DataContainer/Node/Graph}
     */
    Graph.prototype.putAll = function (other) {
        other = config.assertInstanceOf(Graph, other);
        other.nodes.forEach(function (nodeValue, node) {
            var thisNode = this.nodes.put(nodeValue);

            thisNode.next.putAll(node.next);
            thisNode.prev.putAll(node.prev);
        }, this);
    };

    /**
     * @param {mixed} nodeValue
     * @return {void}
     * @throws {Error} if graph does not contain vertice
     */
    Graph.prototype.remove = function (nodeValue) {
        this.nodes.forEach(function (index, node) {
            if (node.hasSibling(nodeValue)) {
                node.removeSibling(nodeValue);
            }
        });

        if (this.nodes.has(nodeValue)) {
            this.nodes.remove(nodeValue);
        }

        this.vertices.remove(nodeValue);
    };

    return Graph;

}));
