/**
 * @license Copyright (c) 2013, CKSource - Frederico Knabben
 * For licensing, see LICENSE
 */

"use strict";

/* global describe: false, it: false */

var assert = require("assert"),
    config = require(global.MODULE_ROOT + "/libraries/modules/configuration-processor").configurationProcessor;

describe("JsLoader/ConfigurationProcessor/modules/configuration-processor", function () {

    describe("#assertArray", function () {

        it("validates input agains an array", function () {
            assert.deepEqual([0, 0, 1], config.assertArray([0, 0, 1]));
        });

    });

    describe("#assertArrayDefault", function () {

        it("validates input agains an array", function () {
            assert.deepEqual([0, 0, 1], config.assertArrayDefault([0, 0, 1], null));
        });

        it("validates that non-empty array is returned", function () {
            assert.deepEqual([0, 1, 1], config.assertArrayDefault([0, 0, 1], [0, 1, 1]));
        });

    });

    describe("#assertArrayEach", function () {

        it("checks input type", function () {
            var input;

            input = [{
                key1: "value1",
                key2: 12
            }];

            assert.strictEqual(input, config.assertArrayEach(config.expectSchema({
                key1: config.expectStringRegExp(/^value1$/),
                key2: config.expectInt()
            }), input));
        });

    });

    describe("#assertArrayHasValue", function () {

        it("asserts that array does not have value", function () {
            var input;

            input = ["value1"];

            assert.throws(function () {
                config.assertArrayHasValue("value2", input);
            }, config.AssertionFailureError);
        });

        it("asserts that array has value", function () {
            var input;

            input = ["value1"];
            assert.strictEqual(input, config.assertArrayHasValue("value1", input));
        });

    });

    describe("#assertArrayLength", function () {

        it("validates input agains an array", function () {
            assert.deepEqual([0, 1, 0], config.assertArrayLength(3, [0, 1, 0]));
        });

    });

    describe("#assertArrayLengthRange", function () {

        it("validates input agains an array", function () {
            assert.deepEqual([0, 1, 1], config.assertArrayLengthRange(1, 3, [0, 1, 1]));
        });

    });

    describe("#assertBooleanDefault", function () {

        it("checks input type", function () {
            assert.strictEqual(true, config.assertBooleanDefault(true, null));
        });

        it("checks edge case", function () {
            assert.strictEqual(false, config.assertBooleanDefault(true, false));
        });

    });

    describe("#assertEmpty", function () {

        it("accepts empty array", function () {
            var input = [];

            assert.strictEqual(input, config.assertEmpty(input));
        });

        it("accepts empty object", function () {
            var input = {};

            assert.strictEqual(input, config.assertEmpty(input));
        });

        it("accepts empty string", function () {
            var input = "";

            assert.strictEqual(input, config.assertEmpty(input));
        });

        it("accepts null", function () {
            var input = null;

            assert.strictEqual(input, config.assertEmpty(input));
        });

        it("accepts zeroes", function () {
            assert.strictEqual(+0, config.assertEmpty(+0));
            assert.strictEqual(-0, config.assertEmpty(-0));
            assert.strictEqual(0, config.assertEmpty(0));
        });

        it("throws an error if item is not empty", function () {
            assert.throws(function () {
                config.assertEmpty(1);
            }, config.AssertionFailureError);
        });

    });

    describe("#assertEnum", function () {

        it("validates input against an enum", function () {
            assert.strictEqual("foo", config.assertEnum(["foo", "bar", "baz"], "foo"));
        });

        it("throws an error if enum does not contain item", function () {
            assert.throws(function () {
                config.assertEnum(["foo", "bar", "baz"], "not-in-the-enum");
            }, config.AssertionFailureError);
        });

    });

    describe("#assertEnumDefault", function () {

        it("validates input against an enum", function () {
            assert.strictEqual("foo", config.assertEnumDefault(["foo", "bar", "baz"], "bar", "foo"));
        });

        it("fails if given value is not inside an enum", function () {
            assert.throws(function () {
                assert.strictEqual("bar", config.assertEnumDefault(["foo", "bar", "baz"], "bar", "not-in-the-enum"));
            }, config.AssertionFailureError);
        });

        it("picks the default value on empty default value", function () {
            assert.strictEqual("bar", config.assertEnumDefault(["foo", "bar", "baz"], "bar", null));
        });

        it("fails if default value is not inside an enum", function () {
            assert.throws(function () {
                config.assertEnumDefault(["foo", "bar", "baz"], "also-not-in-the-enum", "not-in-the-enum");
            }, config.AssertionFailureError);
        });

    });

    describe("#assertFalsy", function () {

        it("accepts falsy values", function () {
            assert.strictEqual(0, config.assertFalsy(0));
        });

        it("throws an error if item is not falsy", function () {
            assert.throws(function () {
                config.assertFalsy({});
            }, config.AssertionFailureError);
        });

    });

    describe("#assertGroup", function () {

        it("mixes several expectations", function () {
            assert.strictEqual("foo", config.assertGroup([
                config.expectNotEmpty(),
                config.expectString()
            ], "foo"));
        });

    });

    describe("#assertGroupDefault", function () {

        it("picks default value", function () {
            assert.strictEqual("foo", config.assertGroupDefault([
                config.expectNotEmpty(),
                config.expectString()
            ], "foo", null));
        });

        it("picks input value", function () {
            assert.strictEqual("bar", config.assertGroupDefault([
                config.expectNotEmpty(),
                config.expectString()
            ], "foo", "bar"));
        });

    });

    describe("#assertGroupOr", function () {

        it("is satisfied if any expectation is fulfilled", function () {
            assert.strictEqual("foo", config.assertGroupOr([
                config.expectString(),
                config.expectObject()
            ], "foo"));
        });

        it("aggregates exceptions", function () {
            assert.throws(function () {
                assert.strictEqual("foo", config.assertGroupOr([
                    config.expectInt(),
                    config.expectObject()
                ], "foo"));
            }, config.AssertionFailureError);
        });

        it("does not intercept exceptions not connected with config", function () {
            assert.throws(function () {
                assert.strictEqual("foo", config.assertGroupOr([
                    function () {
                        throw new Error("This one should not be caught.");
                    }
                ], "foo"));
            }, Error);
        });

    });

    describe("#assertGroupOrDefault", function () {

        it("picks default value", function () {
            assert.strictEqual("foo", config.assertGroupOrDefault([
                config.expectObject(),
                config.expectString()
            ], "foo", null));
        });

        it("picks input value", function () {
            assert.strictEqual("bar", config.assertGroupOrDefault([
                config.expectObject(),
                config.expectString()
            ], "foo", "bar"));
        });

    });

    describe("#assertInstanceOf", function () {

        it("checks inheritance chain", function () {
            var Parent = function () {},
                Child = function () {},
                child;

            Child.prototype = Object.create(Parent.prototype);
            child = new Child();

            assert.strictEqual(child, config.assertInstanceOf(Parent, child));
        });

        it("fails when item is not an instance of parent", function () {
            assert.throws(function () {
                config.assertInstanceOf(function () {}, function () {});
            }, config.AssertionFailureError);
        });

        it("fails when parent is not a function", function () {
            assert.throws(function () {
                config.assertInstanceOf({}, function () {});
            }, config.AssertionFailureError);
        });

    });

    describe("#assertInt", function () {

        it("checks input type", function () {
            assert.strictEqual(5, config.assertInt(5));
        });

        it("rejects floating point numbers", function () {
            assert.throws(function () {
                config.assertInt(5.5);
            }, config.AssertionFailureError);
        });

    });

    describe("#assertIntDefault", function () {

        it("checks edge case", function () {
            assert.strictEqual(0, config.assertIntDefault(3, 0));
        });

        it("picks default value", function () {
            assert.strictEqual(5, config.assertIntDefault(5, null));
        });

    });

    describe("#assertIntRange", function () {

        it("validates input against a maximum offset", function () {
            assert.strictEqual(5, config.assertIntRange(0, 10, 5));
        });

    });

    describe("#assertIntRangeDefault", function () {

        it("picks default value", function () {
            assert.strictEqual(6, config.assertIntRangeDefault(0, 10, 6, null));
        });

        it("picks max range input value", function () {
            assert.strictEqual(0, config.assertIntRangeDefault(0, 10, 6, 0));
        });

        it("picks min range input value", function () {
            assert.strictEqual(10, config.assertIntRangeDefault(0, 10, 6, 10));
        });

    });

    describe("#assertNumber", function () {

        it("validates input against an enum", function () {
            assert.strictEqual(3.5, config.assertNumber(3.5));
        });

    });

    describe("#assertNumberRange", function () {

        it("validates input against a range", function () {
            assert.strictEqual(3.5, config.assertNumberRange(0, 5, 3.5));
        });

    });

    describe("#assertMap", function () {

        it("checks input type", function () {
            var input;

            input = {
                key1: "value1",
                key2: "value2"
            };

            assert.strictEqual(input, config.assertMap(config.expectString(), input));
        });

        it("checks embedded map", function () {
            var input;

            input = {
                key1: {
                    key1n1: {
                        keyXn1: "key1.key1n1.value_keyXn1",
                        keyXn2: 1
                    }
                },
                key2: {
                    key2n1: {
                        keyXn1: "key2.key2n1.value_keyXn1",
                        keyXn2: 1
                    }
                }
            };

            assert.strictEqual(input, config.assertMap(config.expectMap(config.expectSchema({
                keyXn1: config.expectString(),
                keyXn2: config.expectInt()
            })), input));
        });

    });

    describe("#assertNumber", function () {

        it("does not accept Infinity", function () {
            assert.throws(function () {
                config.assertNumber(Infinity);
            }, assert.AssertionFailureError);
        });

        it("does not accept NaN", function () {
            assert.throws(function () {
                config.assertNumber(NaN);
            }, assert.AssertionFailureError);
        });

    });

    describe("#assertNumberDefault", function () {

        it("checks edge case", function () {
            assert.strictEqual(0, config.assertNumberDefault(3.5, 0));
        });

        it("picks default value", function () {
            assert.strictEqual(5.4, config.assertNumberDefault(5.4, null));
        });

    });

    describe("#assertNumberRange", function () {

        it("throws error if number is out of range", function () {
            assert.throws(function () {
                config.assertNumberRange(0.5, 10, 20);
            }, assert.AssertionFailureError);
        });

    });

    describe("#assertNumberRangeDefault", function () {

        it("picks default value", function () {
            assert.strictEqual(7, config.assertNumberRangeDefault(0.5, 10, 7, null));
        });

        it("picks input value", function () {
            assert.strictEqual(8, config.assertNumberRangeDefault(0.5, 10, 7, 8));
        });

    });

    describe("#assertNotEmpty", function () {

        it("checks input type", function () {
            assert.strictEqual(1, config.assertNotEmpty(1));
        });

        it("throws error if item is empty", function () {
            assert.throws(function () {
                config.assertNotEmpty(null);
            }, config.AssertionFailureError);
        });

        it("throws error if item is an empty object", function () {
            assert.throws(function () {
                config.assertNotEmpty({});
            }, config.AssertionFailureError);
        });

    });

    describe("#assertObjectHasKeys", function () {

        it("asserts that object does not have all required keys", function () {
            var input;

            input = {
                key1: "value1",
                key2: "value2"
            };

            assert.throws(function () {
                config.assertObjectHasKeys(["key1", "key2", "key3"], input);
            }, config.AssertionFailureError);
        });

        it("asserts that object has all required keys", function () {
            var input;

            input = {
                key1: "value1",
                key2: "value2",
                key3: "value3"
            };

            assert.deepEqual(input, config.assertObjectHasKeys(["key1", "key2", "key3"], input));
        });

    });

    describe("#assertOptional", function () {

        it("picks input value", function () {
            var input = "testValue";

            assert.equal(input, config.assertOptional(config.expectString(), input));
        });

        it("picks empty value", function () {
            assert.equal(null, config.assertOptional(config.expectString(), null));
        });

    });

    describe("#assertRegExp", function () {

        it("checks input type", function () {
            var input = (/regexp/);

            assert.strictEqual(input, config.assertRegExp(input));
        });

        it("throws error if item is not a regular expression", function () {
            assert.throws(function () {
                config.assertRegExp("not-a-regular-expression");
            }, config.AssertionFailureError);
        });

    });

    describe("#assertSchema", function () {

        it("passess non-configuration related errors", function () {
            assert.throws(function () {
                config.assertSchema({
                    key1: function (item) {
                        throw new Error("This is a custom error.");
                    }
                }, {
                    key1: "key1_value"
                });
            }, Error);
        });

        it("checks default value and input value to match schema", function () {
            assert.throws(function () {
                config.assertSchema({
                    key1: config.expectIntDefault("not-an-integer")
                }, {});
            });
        });

        it("validates input against a plain schema", function () {
            var input,
                regexp = (/foo/),
                schema;

            schema = {
                key1: config.expectArrayDefault([1, 2]),
                key2: config.expectEmpty(),
                key3: config.expectEnum([1, 2]),
                key4: config.expectEnumDefault([1, 2], 1),
                key5: config.expectNumber(),
                key6: config.expectNumberDefault(12),
                key7: config.expectNumberRange(10, 20),
                key8: config.expectNumberRangeDefault(10, 20, 13),
                key9: config.expectIntDefault(3),
                key10: config.expectIntRangeDefault(20, 40, 40),
                key11: config.expectRegExp(),
                key12: config.expectStringDefault("value12"),
                key13: config.expectType("undefined"),
                key14: config.expectTypeDefault("undefined", void 0, null)
            };

            input = {
                key2: void 0,
                key3: 2,
                key5: 6.5,
                key6: null,
                key7: 15,
                key8: 14,
                key10: 23,
                key11: regexp,
            };

            assert.deepEqual({
                key1: [1, 2],
                key2: void 0,
                key3: 2,
                key4: 1,
                key5: 6.5,
                key6: 12,
                key7: 15,
                key8: 14,
                key9: 3,
                key10: 23,
                key11: regexp,
                key12: "value12",
                key13: void 0,
                key14: void 0
            }, config.assertSchema(schema, input));
        });

        it("validates input against a nested schema", function () {
            var input,
                schema;

            schema = {
                key1: config.expectStringExact("value1"),
                key2: config.expectStringRegExp(/^value2$/),
                key3: config.expectSchema({
                    key3n1: config.expectIntRange(10, 20),
                    key3n2: config.expectArray(config.expectGroup([
                        config.expectString(),
                        config.expectNotEmpty()
                    ]))
                }),
                key4: config.expectBoolean(),
                key5: config.expectBooleanDefault(false),
                key6: config.expectStringRegExpDefault((/^value3+$/), "value33"),
                key7: config.expectArrayEach(config.expectSchema({
                    key7n1: config.expectIntRange(10, 20),
                    key7n2: config.expectStringExact("value4"),
                    key7n3: config.expectGroupOr([
                        config.expectArray(),
                        config.expectObject()
                    ])
                }))
            };

            input = {
                key1: "value1",
                key2: "value2",
                key3: {
                    key3n1: 15,
                    key3n2: [
                        "key3n2v1",
                        "key3n2v2"
                    ]
                },
                key4: false,
                key7: [{
                    key7n1: 15,
                    key7n2: "value4",
                    key7n3: {}
                }, {
                    key7n1: 16,
                    key7n2: "value4",
                    key7n3: []
                }]
            };

            assert.deepEqual({
                key1: "value1",
                key2: "value2",
                key3: {
                    key3n1: 15,
                    key3n2: [
                        "key3n2v1",
                        "key3n2v2"
                    ]
                },
                key4: false,
                key5: false,
                key6: "value33",
                key7: [{
                    key7n1: 15,
                    key7n2: "value4",
                    key7n3: {}
                }, {
                    key7n1: 16,
                    key7n2: "value4",
                    key7n3: []
                }]
            }, config.assertSchema(schema, input));
        });

        it("generates errors with full item path", function () {
            var input,
                schema;

            schema = {
                key1: config.expectSchema({
                    key1n1: config.expectSchema({
                        key1n1n1: config.expectString(),
                        key1n1n2: config.expectString()
                    }),
                    key1n2: config.expectSchema({
                        key1n2n1: config.expectInt(),
                        key1n2n2: config.expectSchema({
                            key1n2n2n1: config.expectBoolean()
                        })
                    })
                }),
            };

            input = {
                key1: {
                    key1n1: {
                        key1n1n1: "value_key1n1n1",
                        key1n1n2: "value_key1n1n2"
                    },
                    key1n2: {
                        key1n2n1: 13,
                        key1n2n2: {
                            key1n2n2n1: "it-fails-here"
                        }
                    }
                }
            };

            assert.throws(function () {
                try {
                    config.assertSchema(schema, input);
                } catch (e) {
                    assert.notEqual(-1, e.message.indexOf("key1.key1n2.key1n2n2.key1n2n2n1"));

                    throw e;
                }
            }, config.AssertionFailureError);
        });

    });

    describe("#assertString", function () {

        it("checks input type", function () {
            assert.strictEqual("foo", config.assertString("foo"));
        });

        it("rejects everything but string", function () {
            assert.throws(function () {
                config.assertString(5);
            }, config.AssertionFailureError);
        });

    });

    describe("#assertStringDefault", function () {

        it("picks default value", function () {
            assert.strictEqual("foo", config.assertStringDefault("foo", null));
        });

        it("picks input value", function () {
            assert.strictEqual("bar", config.assertStringDefault("foo", "bar"));
        });

    });

    describe("#assertStringExact", function () {

        it("validates input against a string", function () {
            assert.strictEqual("foo", config.assertStringExact("foo", "foo"));
        });

        it("throws error if other string value given", function () {
            assert.throws(function () {
                config.assertStringExact("foo", "bar");
            }, config.AssertionFailureError);
        });

    });

    describe("#assertStringExactDefault", function () {

        it("picks default value", function () {
            assert.strictEqual("foo", config.assertStringExactDefault("foo", "foo", null));
        });

        it("picks input value", function () {
            assert.strictEqual("foo", config.assertStringExactDefault("foo", "foo", "foo"));
        });

    });

    describe("#assertStringRegExp", function () {

        it("validates input against a string", function () {
            assert.strictEqual("foo", config.assertStringRegExp((/foo/), "foo"));
        });

    });

    describe("#assertStringRegExpDefault", function () {

        it("picks default value", function () {
            assert.strictEqual("foo", config.assertStringRegExpDefault((/fo+/), "foo", null));
        });

        it("picks input value", function () {
            assert.strictEqual("fooo", config.assertStringRegExpDefault((/fo+/), "foo", "fooo"));
        });

        it("fails when default value does not match regular expression", function () {
            assert.throws(function () {
                config.assertStringRegExpDefault((/fo+/), "bar", null);
            }, config.AssertionFailureError);
        });

    });

    describe("#assertType", function () {

        it("checks input type", function () {
            var input = function () {};

            assert.strictEqual(input, config.assertType("function", input));
        });

    });

    describe("#assertTypeDefault", function () {

        it("picks default value", function () {
            var dft = function () {};

            assert.strictEqual(dft, config.assertTypeDefault("function", dft, null));
        });

        it("picks input value", function () {
            var dft = function () {},
                input = function () {};

            assert.strictEqual(input, config.assertTypeDefault("function", dft, input));
        });

    });

    describe("#expectChildOf", function () {

        var Child,
            NotChild,
            Parent;

        Parent = function () {};
        NotChild = function () {};

        Child = function () {
            Parent.call(this);
        };
        Child.prototype = Object.create(Parent.prototype);

        it("rejects object as a child", function () {
            assert.throws(function () {
                config.expectChildOf(Parent)(NotChild);
            }, config.AssertionFailureError);
        });

        it("recognizes object as a child", function () {
            assert.strictEqual(Child, config.expectChildOf(Parent)(Child));
        });

    });

    describe("#expectDefault", function () {

        it("fails on invalid value", function () {
            assert.throws(function () {
                config.expectDefault(config.expectString(), "123")(456);
            }, config.AssertionFailureError);
        });

        it("picks input value", function () {
            var defaultValue = "defaultValue",
                input = "testValue";

            assert.equal(input, config.expectDefault(config.expectString(), defaultValue)(input));
        });

        it("picks default value", function () {
            var defaultValue = "defaultValue";

            assert.equal(defaultValue, config.expectDefault(config.expectString(), defaultValue)());
        });

    });

    describe("#expectFalsy", function () {

        var expectation = config.expectFalsy();

        it("asserts that input is falsy", function () {
            assert.strictEqual(0, expectation(0));
        });

        it("asserts that input is not falsy", function () {
            assert.throws(function () {
                expectation(true);
            }, config.AssertionFailureError);
        });

    });

    describe("#expectFunction", function () {

        it("identifies input as a function", function () {
            var input = function () {};

            assert.deepEqual({
                key1: input
            }, config.assertSchema({
                key1: config.expectFunction()
            }, {
                key1: input
            }));
        });

    });

    describe("#expectGroupDefault", function () {

        it("picks default value", function () {
            assert.deepEqual({
                key1: "value_key1"
            }, config.assertSchema({
                key1: config.expectGroupDefault([
                    config.expectNotEmpty(),
                    config.expectString()
                ], "value_key1")
            }, {}));
        });

        it("picks input value", function () {
            assert.deepEqual({
                key1: "value_key1_custom"
            }, config.assertSchema({
                key1: config.expectGroupDefault([
                    config.expectNotEmpty(),
                    config.expectString()
                ], "value_key1")
            }, {
                key1: "value_key1_custom"
            }));
        });

    });

    describe("#expectGroupOrDefault", function () {

        it("picks default value", function () {
            assert.deepEqual({
                key1: "value_key1"
            }, config.assertSchema({
                key1: config.expectGroupOrDefault([
                    config.expectObject(),
                    config.expectString()
                ], "value_key1")
            }, {}));
        });

        it("picks input value", function () {
            assert.deepEqual({
                key1: "value_key1_custom"
            }, config.assertSchema({
                key1: config.expectGroupOrDefault([
                    config.expectObject(),
                    config.expectString()
                ], "value_key1")
            }, {
                key1: "value_key1_custom"
            }));
        });

    });

    describe("#expectInstanceOf", function () {

        it("checks inheritance chain", function () {
            var Parent = function () {},
                Child = function () {},
                child;

            Child.prototype = Object.create(Parent.prototype);
            child = new Child();

            assert.deepEqual({
                key1: child
            }, config.assertSchema({
                key1: config.expectInstanceOf(Parent)
            }, {
                key1: child
            }));
        });

    });

    describe("#expectInstanceOfDefault", function () {

        it("picks default value", function () {
            var Parent = function () {},
                Child = function () {},
                correct,
                child1,
                child2,
                input,
                output;

            Child.prototype = Object.create(Parent.prototype);
            child1 = new Child();
            child2 = new Child();

            input = {
                key1: void 0
            };

            correct = {
                key1: child1
            };

            output = config.assertSchema({
                key1: config.expectInstanceOfDefault(Parent, child1)
            }, input);

            assert.deepEqual(output, correct);
            assert.strictEqual(output.key1, correct.key1);
        });

        it("picks input value", function () {
            var Parent = function () {},
                Child = function () {},
                correct,
                child1,
                child2,
                input,
                output;

            Child.prototype = Object.create(Parent.prototype);
            child1 = new Child();
            child2 = new Child();

            input = {
                key1: child2
            };

            correct = {
                key1: child2
            };

            output = config.assertSchema({
                key1: config.expectInstanceOfDefault(Parent, child1)
            }, input);

            assert.deepEqual(output, correct);
            assert.strictEqual(output.key1, correct.key1);
        });

    });

    describe("#expectNotEmptyMap", function () {

        var expectation = config.expectNotEmptyMap(config.expectString());

        it("allows not empty map", function () {
            var input;

            input = {
                testIndex: "testValue"
            };
            assert.deepEqual(input, expectation(input));
        });

        it("fails at empty map", function () {
            assert.throws(function () {
                expectation({});
            }, config.AssertionFailureError);
        });

    });

    describe("#expectObjectHasKeys", function () {

        var expectation = config.expectObjectHasKeys(["key1", "key2", "key3"]);

        it("asserts that object does not have all required keys", function () {
            var expectation,
                input;

            input = {
                key1: "value1",
                key2: "value2"
            };

            assert.throws(function () {
                expectation(input);
            }, config.AssertionFailureError);
        });

        it("asserts that object has all required keys", function () {
            var input;

            input = {
                key1: "value1",
                key2: "value2",
                key3: "value3"
            };

            assert.deepEqual(input, expectation(input));
        });

    });

    describe("#expectOptional", function () {

        it("fails on invalid value", function () {
            assert.throws(function () {
                config.expectOptional(config.expectString())(123);
            }, config.AssertionFailureError);
        });

        it("picks input value", function () {
            var input = "testValue";

            assert.equal(input, config.expectOptional(config.expectString())(input));
        });

        it("picks empty value", function () {
            assert.equal(null, config.expectOptional(config.expectString())(null));
        });

    });

    describe("#expectTruthy", function () {

        var expectation = config.expectTruthy();

        it("asserts that input is not truthy", function () {
            assert.throws(function () {
                expectation(false);
            }, config.AssertionFailureError);
        });

        it("asserts that input is truthy", function () {
            assert.strictEqual(5, expectation(5));
        });

    });

});
