/**
 * @license Copyright (c) 2013, CKSource - Frederico Knabben
 * For licensing, see LICENSE
 */

"use strict";

/* global define: false */

(function (factory) {

    if ("function" === typeof define) {
        define([
            "configuration-processor/Error/AssertionFailure",
            "configuration-processor/Error/AssertionFailure/Aggregated",
            "configuration-processor/Error/AssertionFailure/Traceable"
        ], factory);
    } else {
        module.exports = factory.apply(null, [
            require(__dirname + "/Error/AssertionFailure"),
            require(__dirname + "/Error/AssertionFailure/Aggregated"),
            require(__dirname + "/Error/AssertionFailure/Traceable")
        ]);
    }

}(function (AssertionFailureError, AssertionFailureAggregatedError, AssertionFailureTraceableError) {

    var ConfigurationProcessor, // constructor, function
        trace; // private, function

    /**
     * @access private
     * @param {JsLoader/ConfigurationProcessor/modules/configurationProcessor} config
     * @param {string} property
     * @param {function} callback
     * @return {mixed}
     * @throws {JsLoader/ConfigurationProcessor/Error/AssertionFailure}
     */
    trace = function (config, property, callback) {
        try {
            return callback(config, property);
        } catch (e) {
            if (e instanceof AssertionFailureTraceableError) {
                e.trace.unshift(property);
            } else if (e instanceof AssertionFailureError) {
                throw new AssertionFailureTraceableError(e.message, e.item, [property]);
            }

            throw e;
        }
    };

    /**
     * @constructor
     */
    ConfigurationProcessor = function () {};

    /**
     * @param {mixed} item
     * @return {array}
     * @throws {JsLoader/ConfigurationProcessor/Error/AssertionFailure}
     */
    ConfigurationProcessor.prototype.assertArray = function (item) {
        if (!this.isArray(item)) {
            throw new AssertionFailureError("Item is not an array", item);
        }

        return item;
    };

    /**
     * @param {mixed} dft default value
     * @param {mixed} item
     * @return {array}
     * @throws {JsLoader/ConfigurationProcessor/Error/AssertionFailure}
     */
    ConfigurationProcessor.prototype.assertArrayDefault = function (dft, item) {
        if (this.isFalsy(item)) {
            return this.assertArray(dft);
        }

        return this.assertArray(item);
    };

    /**
     * @param {function} expectation
     * @param {mixed} item
     * @return {array}
     * @throws {JsLoader/ConfigurationProcessor/Error/AssertionFailure}
     */
    ConfigurationProcessor.prototype.assertArrayEach = function (expectation, item) {
        var doAssert,
            i;

        expectation = this.assertFunction(expectation);
        item = this.assertArray(item);

        doAssert = function (config, property) {
            return expectation(item[property]);
        };

        for (i = 0; i < item.length; i += 1) {
            item[i] = trace(this, i, doAssert);
        }

        return item;
    };

    /**
     * @param {mixed} value
     * @param {mixed} item
     * @return {array}
     * @throws {JsLoader/ConfigurationProcessor/Error/AssertionFailure}
     */
    ConfigurationProcessor.prototype.assertArrayHasValue = function (value, item) {
        var i;

        item = this.assertArray(item);

        for (i = 0; i < item.length; i += 1) {
            if (item[i] === value) {
                return item;
            }
        }

        throw new AssertionFailureError("Item does not contain: " + item.toString(), item);
    };

    /**
     * @param {function} expectation
     * @param {mixed} dft default value
     * @param {mixed} item
     * @return {array}
     * @throws {JsLoader/ConfigurationProcessor/Error/AssertionFailure}
     */
    ConfigurationProcessor.prototype.assertArrayLength = function (length, item) {
        return this.assertArrayLengthRange(length, length, item);
    };

    /**
     * @param {int} lengthMin
     * @param {int} lengthMax
     * @param {mixed} item
     * @return {array}
     * @throws {JsLoader/ConfigurationProcessor/Error/AssertionFailure}
     */
    ConfigurationProcessor.prototype.assertArrayLengthRange = function (lengthMin, lengthMax, item) {
        item = this.assertArray(item);

        this.assertIntRange(lengthMin, lengthMax, item.length);

        return item;
    };

    /**
     * @param {mixed} item
     * @return {bool}
     */
    ConfigurationProcessor.prototype.assertBoolean = function (item) {
        return this.assertType("boolean", item);
    };

    /**
     * @param {mixed} dft default value
     * @param {mixed} item
     * @return {mixed}
     * @throws {JsLoader/ConfigurationProcessor/Error/AssertionFailure}
     */
    ConfigurationProcessor.prototype.assertBooleanDefault = function (dft, item) {
        if (this.isFalsy(item) && (false !== item)) {
            return this.assertBoolean(dft);
        }

        return this.assertBoolean(item);
    };

    /**
     * @param {function} Parent
     * @param {mixed} Item
     * @return {function}
     * @throws {JsLoader/ConfigurationProcessor/Error/AssertionFailure}
     */
    ConfigurationProcessor.prototype.assertChildOf = function (Parent, Item) {
        Parent = this.assertFunction(Parent);
        this.assertInstanceOf(Parent, Item.prototype);

        return this.assertFunction(Item);
    };

    /**
     * @param {function} expectation
     * @param {mixed} dft
     * @param {mixed} item
     * @return {mixed}
     * @throws {JsLoader/ConfigurationProcessor/Error/AssertionFailure}
     */
    ConfigurationProcessor.prototype.assertDefault = function (expectation, dft, item) {
        if (this.isEmpty(item)) {
            return expectation(dft);
        }

        return expectation(item);
    };

    /**
     * @param {mixed} item
     * @return {mixed}
     * @throws {JsLoader/ConfigurationProcessor/Error/AssertionFailure}
     */
    ConfigurationProcessor.prototype.assertEmpty = function (item) {
        if (this.isEmpty(item)) {
            return item;
        }

        throw new AssertionFailureError("Item is not empty", item);
    };

    /**
     * @param {array} enumeration
     * @param {mixed} item
     * @return {mixed}
     * @throws {JsLoader/ConfigurationProcessor/Error/AssertionFailure}
     */
    ConfigurationProcessor.prototype.assertEnum = function (enumeration, item) {
        var dump,
            i;

        enumeration = this.assertArray(enumeration);

        for (i = 0; i < enumeration.length; i += 1) {
            if (item === enumeration[i]) {
                return item;
            }
        }

        dump = enumeration.join(", ");
        throw new AssertionFailureError("Enumeration [" + dump + "] does not contain item", item);
    };

    /**
     * @param {array} enumeration
     * @param {mixed} dft default value
     * @param {mixed} item
     * @return {mixed}
     * @throws {JsLoader/ConfigurationProcessor/Error/AssertionFailure}
     */
    ConfigurationProcessor.prototype.assertEnumDefault = function (enumeration, dft, item) {
        if (this.isFalsy(item)) {
            return this.assertEnum(enumeration, dft);
        }

        return this.assertEnum(enumeration, item);
    };

    /**
     * @param {mixed} item
     * @return {mixed}
     * @throws {JsLoader/ConfigurationProcessor/Error/AssertionFailure}
     */
    ConfigurationProcessor.prototype.assertFalsy = function (item) {
        if (this.isFalsy(item)) {
            return item;
        }

        throw new AssertionFailureError("Item is not falsy", item);
    };

    /**
     * @param {mixed} item
     * @return {function}
     * @throws {JsLoader/ConfigurationProcessor/Error/AssertionFailure}
     */
    ConfigurationProcessor.prototype.assertFunction = function (item) {
        return this.assertType("function", item);
    };

    /**
     * @param {array} expectations
     * @param {mixed} item
     * @return {mixed}
     */
    ConfigurationProcessor.prototype.assertGroup = function (expectations, item) {
        var expectation,
            i;

        expectations = this.assertArray(expectations);
        for (i = 0; i < expectations.length; i += 1) {
            expectation = this.assertFunction(expectations[i]);
            item = expectation(item);
        }

        return item;
    };

    /**
     * @param {array} expectations
     * @param {mixed} dft default value
     * @param {mixed} item
     * @return {mixed}
     * @throws {JsLoader/ConfigurationProcessor/Error/AssertionFailure}
     */
    ConfigurationProcessor.prototype.assertGroupDefault = function (expectations, dft, item) {
        if (this.isFalsy(item)) {
            return this.assertGroup(expectations, dft);
        }

        return this.assertGroup(expectations, item);
    };

    /**
     * @param {array} expectations
     * @param {mixed} item
     * @return {mixed}
     * @throws {JsLoader/ConfigurationProcessor/Error/AssertionFailure}
     */
    ConfigurationProcessor.prototype.assertGroupOr = function (expectations, item) {
        var err = new AssertionFailureAggregatedError("Errors occurred when traversing group of assertions", item),
            expectation,
            i;

        expectations = this.assertArray(expectations);
        for (i = 0; i < expectations.length; i += 1) {
            expectation = this.assertFunction(expectations[i]);
            try {
                return expectation(item);
            } catch (e) {
                if (!this.isAssertionFailureException(e)) {
                    throw e;
                }

                err.add(e);
            }
        }

        throw err;
    };

    /**
     * @param {array} expectations
     * @param {mixed} dft default value
     * @param {mixed} item
     * @return {mixed}
     * @throws {JsLoader/ConfigurationProcessor/Error/AssertionFailure}
     */
    ConfigurationProcessor.prototype.assertGroupOrDefault = function (expectations, dft, item) {
        if (this.isFalsy(item)) {
            return this.assertGroupOr(expectations, dft);
        }

        return this.assertGroupOr(expectations, item);
    };

    /**
     * @param {function} Parent
     * @param {mixed} item
     * @return {function}
     * @throws {JsLoader/ConfigurationProcessor/Error/AssertionFailure}
     */
    ConfigurationProcessor.prototype.assertInstanceOf = function (Parent, item) {
        Parent = this.assertFunction(Parent);

        if (item instanceof Parent) {
            return item;
        }

        throw new AssertionFailureError("Item is not an instance of " + Parent.name, item);
    };

    /**
     * @param {function} Parent
     * @param {mixed} dft
     * @param {mixed} item
     * @return {function}
     * @throws {JsLoader/ConfigurationProcessor/Error/AssertionFailure}
     */
    ConfigurationProcessor.prototype.assertInstanceOfDefault = function (Parent, dft, item) {
        if (this.isFalsy(item)) {
            return this.assertInstanceOf(Parent, dft);
        }

        return this.assertInstanceOf(Parent, item);
    };

    /**
     * @param {mixed} item
     * @return {int}
     * @throws {JsLoader/ConfigurationProcessor/Error/AssertionFailure}
     */
    ConfigurationProcessor.prototype.assertInt = function (item) {
        item = this.assertNumber(item);
        if (parseFloat(item) === parseInt(item, 10)) {
            return item;
        }

        throw new AssertionFailureError("Item is not an integer", item);
    };

    /**
     * @param {int} dft default value
     * @param {mixed} item
     * @return {int}
     * @throws {JsLoader/ConfigurationProcessor/Error/AssertionFailure}
     */
    ConfigurationProcessor.prototype.assertIntDefault = function (dft, item) {
        if (this.isFalsy(item) && (0 !== item)) {
            return this.assertInt(dft);
        }

        return this.assertInt(item);
    };

    /**
     * @param {number} rangeMin
     * @param {number} rangeMax
     * @param {mixed} item
     * @return {int}
     * @throws {JsLoader/ConfigurationProcessor/Error/AssertionFailure}
     */
    ConfigurationProcessor.prototype.assertIntRange = function (rangeMin, rangeMax, item) {
        item = this.assertInt(item);

        return this.assertNumberRange(rangeMin, rangeMax, item);
    };

    /**
     * @param {number} rangeMin
     * @param {number} rangeMax
     * @param {int} dft
     * @return {function}
     * @throws {JsLoader/ConfigurationProcessor/Error/AssertionFailure}
     */
    ConfigurationProcessor.prototype.assertIntRangeDefault = function (rangeMin, rangeMax, dft, item) {
        if (this.isFalsy(item) && (0 !== item)) {
            return this.assertIntRange(rangeMin, rangeMax, dft);
        }

        return this.assertIntRange(rangeMin, rangeMax, item);
    };

    /**
     * @param {function} expectation
     * @param {mixed} item
     * @return {object}
     * @throws {JsLoader/ConfigurationProcessor/Error/AssertionFailure}
     */
    ConfigurationProcessor.prototype.assertMap = function (expectation, item) {
        var doAssert,
            property;

        expectation = this.assertFunction(expectation);
        item = this.assertObject(item);

        doAssert = function (config, property) {
            return expectation(item[property]);
        };

        for (property in item) {
            if (item.hasOwnProperty(property)) {
                item[property] = trace(this, property, doAssert);
            }
        }

        return item;
    };

    /**
     * @param {mixed} item
     * @return {float}
     * @throws {JsLoader/ConfigurationProcessor/Error/AssertionFailure}
     */
    ConfigurationProcessor.prototype.assertNumber = function (item) {
        item = this.assertType("number", item);

        if (isNaN(item)) {
            throw new AssertionFailureError("Item is NaN", item);
        }

        if (!isFinite(item)) {
            // NaN is not finite, thus checking for infinity should be done after
            // checking for NaN
            throw new AssertionFailureError("Item is an infinity", item);
        }

        return item;
    };

    /**
     * @param {mixed} dft default value
     * @param {mixed} item
     * @return {float}
     * @throws {JsLoader/ConfigurationProcessor/Error/AssertionFailure}
     */
    ConfigurationProcessor.prototype.assertNumberDefault = function (dft, item) {
        if (this.isFalsy(item) && (0 !== item)) {
            return this.assertNumber(dft);
        }

        return this.assertNumber(item);
    };

    /**
     * @param {number} rangeMin
     * @param {number} rangeMax
     * @param {mixed} item
     * @return {float}
     * @throws {JsLoader/ConfigurationProcessor/Error/AssertionFailure}
     */
    ConfigurationProcessor.prototype.assertNumberRange = function (rangeMin, rangeMax, item) {
        item = this.assertNumber(item);
        rangeMin = this.assertNumber(rangeMin);
        rangeMax = this.assertNumber(rangeMax);

        if (item < rangeMin || item > rangeMax) {
            throw new AssertionFailureError("Item is not in range: [" + rangeMin + ", " + rangeMax + "]", item);
        }

        return item;
    };

    /**
     * @param {number} rangeMin
     * @param {number} rangeMax
     * @param {mixed} dft default value
     * @param {mixed} item
     * @return {mixed}
     * @throws {JsLoader/ConfigurationProcessor/Error/AssertionFailure}
     */
    ConfigurationProcessor.prototype.assertNumberRangeDefault = function (rangeMin, rangeMax, dft, item) {
        if (this.isFalsy(item)) {
            return this.assertNumberRange(rangeMin, rangeMax, dft);
        }

        return this.assertNumberRange(rangeMin, rangeMax, item);
    };

    /**
     * @param {mixed} item
     * @return {mixed}
     * @throws {JsLoader/ConfigurationProcessor/Error/AssertionFailure}
     */
    ConfigurationProcessor.prototype.assertNotEmpty = function (item) {
        if (this.isEmpty(item)) {
            throw new AssertionFailureError("Item is empty", item);
        }

        return item;
    };

    /**
     * @param {function} expectation
     * @param {mixed} item
     * @return {object}
     * @throws {JsLoader/ConfigurationProcessor/Error/AssertionFailure}
     */
    ConfigurationProcessor.prototype.assertNotEmptyMap = function (expectation, item) {
        return this.assertGroup([
            this.expectNotEmpty(),
            this.expectMap(expectation)
        ], item);
    };

    /**
     * @param {mixed} item
     * @return {mixed}
     * @throws {JsLoader/ConfigurationProcessor/Error/AssertionFailure}
     */
    ConfigurationProcessor.prototype.assertObject = function (item) {
        return this.assertType("object", item);
    };

    /**
     * @param {array} keys
     * @param {mixed} item
     * @param {function} expectation
     * @return {mixed}
     * @throws {JsLoader/ConfigurationProcessor/Error/AssertionFailure}
     */
    ConfigurationProcessor.prototype.assertObjectHasKeys = function (keys, item) {
        var i;

        keys = this.assertArray(keys);
        item = this.assertObject(item);

        for (i = 0; i < keys.length; i += 1) {
            if (!item.hasOwnProperty(keys[i])) {
                throw new AssertionFailureError("Object does not contain key: " + item, item);
            }
        }

        return item;
    };

    /**
     * @param {function} expectation
     * @param {mixed} item
     * @return {mixed}
     * @throws {JsLoader/ConfigurationProcessor/Error/AssertionFailure}
     */
    ConfigurationProcessor.prototype.assertOptional = function (expectation, item) {
        if (this.isEmpty(item)) {
            return item;
        }

        return expectation(item);
    };

    /**
     * @param {mixed} item
     * @param {function} expectation
     * @return {mixed}
     * @throws {JsLoader/ConfigurationProcessor/Error/AssertionFailure}
     */
    ConfigurationProcessor.prototype.assertRegExp = function (item) {
        return this.assertInstanceOf(RegExp, item);
    };

    /**
     * @param {object} schema
     * @param {object} item
     * @return {mixed}
     */
    ConfigurationProcessor.prototype.assertSchema = function (schema, item) {
        var doAssert,
            property;

        item = this.assertObject(item);
        schema = this.assertObject(schema);

        doAssert = function (config, property) {
            schema[property] = config.assertFunction(schema[property]);

            return schema[property](item[property]);
        };

        for (property in schema) {
            if (schema.hasOwnProperty(property)) {
                item[property] = trace(this, property, doAssert);
            }
        }

        return item;
    };

    /**
     * @param {mixed} item
     * @return {string}
     * @throws {JsLoader/ConfigurationProcessor/Error/AssertionFailure}
     * @see JsLoader/ConfigurationProcessor/modules/configurationProcessor~assertType
     */
    ConfigurationProcessor.prototype.assertString = function (item) {
        // assertType uses assertString internally, thus assertType cannot be used
        // here
        if ("string" === typeof item) {
            return item;
        }

        throw new AssertionFailureError("Item is not a string", item);
    };

    /**
     * @param {string} dft default value
     * @param {mixed} item
     * @return {string}
     * @throws {JsLoader/ConfigurationProcessor/Error/AssertionFailure}
     */
    ConfigurationProcessor.prototype.assertStringDefault = function (dft, item) {
        if (this.isFalsy(item)) {
            return this.assertString(dft);
        }

        return this.assertString(item);
    };

    /**
     * @param {string} pattern
     * @param {mixed} item
     * @return {mixed}
     * @throws {JsLoader/ConfigurationProcessor/Error/AssertionFailure}
     */
    ConfigurationProcessor.prototype.assertStringExact = function (pattern, item) {
        item = this.assertString(item);
        pattern = this.assertString(pattern);

        if (item === pattern) {
            return item;
        }

        throw new AssertionFailureError('Item is not EXACTLY a string with value of "' + pattern + '"', item);
    };

    /**
     * @param {mixed} item
     * @param {mixed} dft default value
     * @param {string} pattern
     * @return {mixed}
     * @throws {JsLoader/ConfigurationProcessor/Error/AssertionFailure}
     */
    ConfigurationProcessor.prototype.assertStringExactDefault = function (pattern, dft, item) {
        if (this.isFalsy(item)) {
            return this.assertStringExact(pattern, dft);
        }

        return this.assertStringExact(pattern, item);
    };

    /**
     * @param {RegExp} regexp
     * @param {mixed} item
     * @return {mixed}
     * @throws {JsLoader/ConfigurationProcessor/Error/AssertionFailure}
     */
    ConfigurationProcessor.prototype.assertStringRegExp = function (regexp, item) {
        item = this.assertString(item);
        regexp = this.assertRegExp(regexp);

        if (regexp.test(item)) {
            return item;
        }

        throw new AssertionFailureError("Item does not match regular expression " + regexp.toString(), item);
    };

    /**
     * @param {mixed} item
     * @param {mixed} dft default value
     * @param {RegExp} regexp
     * @return {mixed}
     * @throws {JsLoader/ConfigurationProcessor/Error/AssertionFailure}
     */
    ConfigurationProcessor.prototype.assertStringRegExpDefault = function (regexp, dft, item) {
        if (this.isFalsy(item)) {
            return this.assertStringRegExp(regexp, dft);
        }

        return this.assertStringRegExp(regexp, item);
    };

    /**
     * @param {mixed} item
     * @return {mixed}
     * @throws {JsLoader/ConfigurationProcessor/Error/AssertionFailure}
     */
    ConfigurationProcessor.prototype.assertTruthy = function (item) {
        if (this.isTruthy(item)) {
            return item;
        }

        throw new AssertionFailureError("Item is not truthy", item);
    };

    /**
     * @param {string} type
     * @param {mixed} item
     * @return {mixed}
     * @throws {JsLoader/ConfigurationProcessor/Error/AssertionFailure}
     */
    ConfigurationProcessor.prototype.assertType = function (type, item) {
        type = this.assertString(type);
        if (type !== typeof item) {
            throw new AssertionFailureError("Item type is not '" + type + "'", item);
        }

        return item;
    };

    /**
     * @param {string} type
     * @param {mixed} item
     * @param {mixed} dft default value
     * @return {mixed}
     * @throws {JsLoader/ConfigurationProcessor/Error/AssertionFailure}
     */
    ConfigurationProcessor.prototype.assertTypeDefault = function (type, dft, item) {
        if (this.isFalsy(item)) {
            return this.assertType(type, dft);
        }

        return this.assertType(type, item);
    };

    /**
     * @return {function}
     */
    ConfigurationProcessor.prototype.expectArray = function () {
        return this.assertArray.bind(this);
    };

    /**
     * @param {function} expectation
     * @return {function}
     */
    ConfigurationProcessor.prototype.expectArrayEach = function (expectation) {
        return this.assertArrayEach.bind(this, expectation);
    };

    /**
     * @param {mixed} dft default value
     * @return {function}
     */
    ConfigurationProcessor.prototype.expectArrayDefault = function (dft) {
        return this.assertArrayDefault.bind(this, dft);
    };

    /**
     * @return {function}
     */
    ConfigurationProcessor.prototype.expectBoolean = function () {
        return this.assertBoolean.bind(this);
    };

    /**
     * @param {bool} dft
     * @return {function}
     */
    ConfigurationProcessor.prototype.expectBooleanDefault = function (dft) {
        return this.assertBooleanDefault.bind(this, dft);
    };

    /**
     * @param {function} Parent
     * @return {function}
     */
    ConfigurationProcessor.prototype.expectChildOf = function (Parent) {
        return this.assertChildOf.bind(this, Parent);
    };

    /**
     * @param {bool} dft
     * @return {function}
     */
    ConfigurationProcessor.prototype.expectDefault = function (expectation, dft) {
        return this.assertDefault.bind(this, expectation, dft);
    };

    /**
     * @return {function}
     */
    ConfigurationProcessor.prototype.expectEmpty = function () {
        return this.assertEmpty.bind(this);
    };

    /**
     * @param {array} enumeration
     * @return {function}
     */
    ConfigurationProcessor.prototype.expectEnum = function (enumeration) {
        return this.assertEnum.bind(this, enumeration);
    };

    /**
     * @param {array} enumeration
     * @param {array} dft
     * @return {function}
     */
    ConfigurationProcessor.prototype.expectEnumDefault = function (enumeration, dft) {
        return this.assertEnumDefault.bind(this, enumeration, dft);
    };

    /**
     * @return {function}
     */
    ConfigurationProcessor.prototype.expectFalsy = function () {
        return this.assertFalsy.bind(this);
    };

    /**
     * @return {function}
     */
    ConfigurationProcessor.prototype.expectFunction = function () {
        return this.assertFunction.bind(this);
    };

    /**
     * @param {function} Parent
     * @return {function}
     */
    ConfigurationProcessor.prototype.expectInstanceOf = function (Parent) {
        return this.assertInstanceOf.bind(this, Parent);
    };

    /**
     * @param {function} Parent
     * @param {function} dft
     * @return {function}
     */
    ConfigurationProcessor.prototype.expectInstanceOfDefault = function (Parent, dft) {
        return this.assertInstanceOfDefault.bind(this, Parent, dft);
    };

    /**
     * @return {function}
     */
    ConfigurationProcessor.prototype.expectNumber = function () {
        return this.assertNumber.bind(this);
    };

    /**
     * @param {number} dft
     * @return {function}
     */
    ConfigurationProcessor.prototype.expectNumberDefault = function (dft) {
        return this.assertNumberDefault.bind(this, dft);
    };

    /**
     * @param {number} rangeMin
     * @param {number} rangeMax
     * @return {function}
     */
    ConfigurationProcessor.prototype.expectNumberRange = function (rangeMin, rangeMax) {
        return this.assertNumberRange.bind(this, rangeMin, rangeMax);
    };

    /**
     * @param {number} rangeMin
     * @param {number} rangeMax
     * @param {float} dft
     * @return {function}
     */
    ConfigurationProcessor.prototype.expectNumberRangeDefault = function (rangeMin, rangeMax, dft) {
        return this.assertNumberRangeDefault.bind(this, rangeMin, rangeMax, dft);
    };

    /**
     * @param {array} expectations
     * @return {function}
     */
    ConfigurationProcessor.prototype.expectGroup = function (expectations) {
        return this.assertGroup.bind(this, expectations);
    };

    /**
     * @param {array} expectations
     * @param {mixed} dft
     * @return {function}
     */
    ConfigurationProcessor.prototype.expectGroupDefault = function (expectations, dft) {
        return this.assertGroupDefault.bind(this, expectations, dft);
    };

    /**
     * @param {array} expectations
     * @return {function}
     */
    ConfigurationProcessor.prototype.expectGroupOr = function (expectations) {
        return this.assertGroupOr.bind(this, expectations);
    };

    /**
     * @param {array} expectations
     * @param {mixed} dft
     * @return {function}
     */
    ConfigurationProcessor.prototype.expectGroupOrDefault = function (expectations, dft) {
        return this.assertGroupOrDefault.bind(this, expectations, dft);
    };

    /**
     * @return {function}
     */
    ConfigurationProcessor.prototype.expectInt = function () {
        return this.assertInt.bind(this);
    };

    /**
     * @param {int} dft
     * @return {function}
     */
    ConfigurationProcessor.prototype.expectIntDefault = function (dft) {
        return this.assertIntDefault.bind(this, dft);
    };

    /**
     * @param {number} rangeMin
     * @param {number} rangeMax
     * @return {function}
     */
    ConfigurationProcessor.prototype.expectIntRange = function (rangeMin, rangeMax) {
        return this.assertIntRange.bind(this, rangeMin, rangeMax);
    };

    /**
     * @param {number} rangeMin
     * @param {number} rangeMax
     * @param {int} dft
     * @return {function}
     */
    ConfigurationProcessor.prototype.expectIntRangeDefault = function (rangeMin, rangeMax, dft) {
        return this.assertIntRangeDefault.bind(this, rangeMin, rangeMax, dft);
    };

    /**
     * @param {function} expectation
     * @return {function}
     */
    ConfigurationProcessor.prototype.expectMap = function (expectation) {
        return this.assertMap.bind(this, expectation);
    };

    /**
     * @return {function}
     */
    ConfigurationProcessor.prototype.expectNotEmpty = function () {
        return this.assertNotEmpty.bind(this);
    };

    /**
     * @param {function} expectation
     * @return {function}
     */
    ConfigurationProcessor.prototype.expectNotEmptyMap = function (expectation) {
        return this.assertNotEmptyMap.bind(this, expectation);
    };

    /**
     * @return {function}
     */
    ConfigurationProcessor.prototype.expectObject = function () {
        return this.assertObject.bind(this);
    };

    /**
     * @param {array} keys
     * @return {function}
     */
    ConfigurationProcessor.prototype.expectObjectHasKeys = function (keys) {
        return this.assertObjectHasKeys.bind(this, keys);
    };

    /**
     * @param {function} expectation
     * @return {function}
     */
    ConfigurationProcessor.prototype.expectOptional = function (expectation) {
        return this.assertOptional.bind(this, expectation);
    };

    /**
     * @return {function}
     */
    ConfigurationProcessor.prototype.expectRegExp = function () {
        return this.assertRegExp.bind(this);
    };

    /**
     * @param {object} schema
     * @return {function}
     */
    ConfigurationProcessor.prototype.expectSchema = function (schema) {
        return this.assertSchema.bind(this, schema);
    };

    /**
     * @return {function}
     */
    ConfigurationProcessor.prototype.expectString = function () {
        return this.assertString.bind(this);
    };

    /**
     * @param {string} dft
     * @return {function}
     */
    ConfigurationProcessor.prototype.expectStringDefault = function (dft) {
        return this.assertStringDefault.bind(this, dft);
    };

    /**
     * @param {string} pattern
     * @return {function}
     */
    ConfigurationProcessor.prototype.expectStringExact = function (pattern) {
        return this.assertStringExact.bind(this, pattern);
    };

    /**
     * @param {RegExp} regexp
     * @return {function}
     */
    ConfigurationProcessor.prototype.expectStringRegExp = function (regexp) {
        return this.assertStringRegExp.bind(this, regexp);
    };

    /**
     * @param {RegExp} regexp
     * @param {string} dft
     * @return {function}
     */
    ConfigurationProcessor.prototype.expectStringRegExpDefault = function (regexp, dft) {
        return this.assertStringRegExpDefault.bind(this, regexp, dft);
    };

    /**
     * @return {function}
     */
    ConfigurationProcessor.prototype.expectTruthy = function () {
        return this.assertTruthy.bind(this);
    };

    /**
     * @param {string} type
     * @return {function}
     */
    ConfigurationProcessor.prototype.expectType = function (type) {
        return this.assertType.bind(this, type);
    };

    /**
     * @param {string} type
     * @param {string} dft
     * @return {function}
     */
    ConfigurationProcessor.prototype.expectTypeDefault = function (type, dft) {
        return this.assertTypeDefault.bind(this, type, dft);
    };

    /**
     * @param {mixed} item
     * @return {bool}
     */
    ConfigurationProcessor.prototype.isArray = function (item) {
        return Array.isArray(item);
    };

    /**
     * @param {object} e
     * @return {bool}
     */
    ConfigurationProcessor.prototype.isAssertionFailureException = function (e) {
        return e instanceof AssertionFailureError;
    };

    /**
     * @param {mixed} item
     * @return {bool}
     */
    ConfigurationProcessor.prototype.isEmpty = function (item) {
        if (this.isFalsy(item)) {
            return true;
        }

        if ("object" === typeof item) {
            return Object.keys(item).length < 1;
        }

        return false;
    };

    /**
     * @param {mixed} item
     * @return {bool}
     */
    ConfigurationProcessor.prototype.isFalsy = function (item) {
        return !item;
    };

    /**
     * @param {mixed} item
     * @return {bool}
     */
    ConfigurationProcessor.prototype.isTruthy = function (item) {
        return !!item;
    };

    return ConfigurationProcessor;

}));
