const User = require("../models/UserModel");
const { body,validationResult } = require("express-validator");
const { sanitizeBody } = require("express-validator");
const apiResponse = require("../helpers/apiResponse");
const auth = require("../middlewares/jwt");
var mongoose = require("mongoose");
mongoose.set("useFindAndModify", false);
const { constants } = require("../helpers/constants");
const bcrypt = require("bcrypt");
const utility = require("../helpers/utility");

// Book Schema
function UserData(data) {
	this.id = data._id;
	this.firstName= data.firstName;
	this.lastName = data.lastName;
	this.email = data.email;
	this.createdAt = data.createdAt;
    this.updatedAt = data.updatedAt;
    this.isConfirmed  = data.isConfirmed
}

/**
 * Book List.
 * 
 * @returns {Object}
 */
exports.userList = [
	auth,
	function (req, res) {
		try {
			User.find({'role': {$ne : "admin"}},"_id firstName lastName email mobileNumber panCard role dob isConfirmed createdAt").then((users)=>{
				if(users.length > 0){
					return apiResponse.successResponseWithData(res, "Operation success", users);
				}else{
					return apiResponse.successResponseWithData(res, "Operation success", []);
				}
			});
		} catch (err) {
			//throw error in json response with status 500. 
			return apiResponse.ErrorResponse(res, err);
		}
	}
];

/**
 * Book Detail.
 * 
 * @param {string}      id
 * 
 * @returns {Object}
 */
exports.userDetail = [
	auth,
	function (req, res) {
		console.log(mongoose.Types.ObjectId.isValid(req.params.id))
		if(!mongoose.Types.ObjectId.isValid(req.params.id)){
			return apiResponse.successResponseWithData(res, "Operation success", {});
		}
		try {
			User.findOne({_id: req.params.id},"_id firstName lastName email mobileNumber isConfirmed createdAt").then((user)=>{                
				if(user !== null){
					let userData = new UserData(user);
					return apiResponse.successResponseWithData(res, "Operation success", userData);
				}else{
					return apiResponse.successResponseWithData(res, "Operation success", {});
				}
			});
		} catch (err) {
			//throw error in json response with status 500. 
			return apiResponse.ErrorResponse(res, err);
		}
	}
];


exports.userStore = [
	auth,
	body("firstName").isLength({ min: 1 }).trim().withMessage("First name must be specified.")
		.isAlphanumeric().withMessage("First name has non-alphanumeric characters."),
	body("lastName").isLength({ min: 1 }).trim().withMessage("Last name must be specified.")
		.isAlphanumeric().withMessage("Last name has non-alphanumeric characters."),
	body("email").isLength({ min: 1 }).trim().withMessage("Email must be specified.")
		.isEmail().withMessage("Email must be a valid email address.").custom((value) => {
			return User.findOne({email : value}).then((user) => {
				if (user) {
					return Promise.reject("E-mail already in use");
				}
			});
		}),
	body("mobileNumber").isLength({ min : 1}).trim().withMessage("Mobile number must be specified").isNumeric().withMessage("Mobile Number should be numeric").isLength({min : 10, max : 10}).trim().withMessage("Mobile Number must be 10 digit numeric value."),
	body("password").isLength({ min: 6 }).trim().withMessage("Password must be 6 characters or greater."),
	// Sanitize fields.
	sanitizeBody("firstName").escape(),
	sanitizeBody("lastName").escape(),
	sanitizeBody("email").escape(),
	sanitizeBody("password").escape(),
	(req, res) => {
		try {
			const errors = validationResult(req);
			if (!errors.isEmpty()) {
				return apiResponse.validationErrorWithData(res, "Validation Error.", errors.array());
			}
			else {
				bcrypt.hash(req.body.password,10,function(err, hash) {
					// generate OTP for confirmation
					let otp = utility.randomNumber(4);
					// Create User object with escaped and trimmed data
					var user = new User(
						{
							firstName: req.body.firstName,
							lastName: req.body.lastName,
							email: req.body.email,
							password: hash,
							confirmOTP: otp,
							mobileNumber : req.body.mobileNumber,
							isConfirmed : true
						}
					);
						// Save user.
						user.save(function (err) {
							if (err) { return apiResponse.ErrorResponse(res, err); }
							let userData = {
								_id: user._id,
								firstName: user.firstName,
								lastName: user.lastName,
								email: user.email,
								mobileNumber : user.mobileNumber,
								isConfirmed : user.isConfirmed
							};
							return apiResponse.successResponseWithData(res,"User Created.", userData);
						});
					
				});
			}
		} catch (err) {
			//throw error in json response with status 500. 
			return apiResponse.ErrorResponse(res, err);
		}
	}
];

/**
 * Book update.
 * 
 * @param {string}      title 
 * @param {string}      description
 * @param {string}      isbn
 * 
 * @returns {Object}
 */
exports.userUpdate = [
	auth,
	body("firstName").isLength({ min: 1 }).trim().withMessage("First name must be specified.")
		.isAlphanumeric().withMessage("First name has non-alphanumeric characters."),
	body("lastName").isLength({ min: 1 }).trim().withMessage("Last name must be specified.")
		.isAlphanumeric().withMessage("Last name has non-alphanumeric characters."),
	body("email").isLength({ min: 1 }).trim().withMessage("Email must be specified.")
		.isEmail().withMessage("Email must be a valid email address."),
	body("mobileNumber").isLength({ min : 1}).trim().withMessage("Mobile number must be specified").isNumeric().withMessage("Mobile Number should be numeric").isLength({min : 10, max : 10}).trim().withMessage("Mobile Number must be 10 digit numeric value."),
	// body("password").isLength({ min: 6 }).trim().withMessage("Password must be 6 characters or greater."),
	// Sanitize fields.
	sanitizeBody("firstName").escape(),
	sanitizeBody("lastName").escape(),
	sanitizeBody("email").escape(),
	sanitizeBody("password").escape(),
	(req, res) => {
		try {
			const errors = validationResult(req);
			console.log(req.body)
			if (!errors.isEmpty()) {
				return apiResponse.validationErrorWithData(res, "Validation Error.", errors.array());
			}
			else {
				if(!mongoose.Types.ObjectId.isValid(req.params.id)){
					return apiResponse.validationErrorWithData(res, "Invalid Error.", "Invalid ID");
				}else{
					User.findById(req.params.id, function (err, foundUser) {
						if(foundUser === null){
							return apiResponse.notFoundResponse(res,"User not exists with this id");
						}else{

							var user = new User(
								{
									firstName: req.body.firstName,
									lastName: req.body.lastName,
									email: req.body.email,
									mobileNumber : req.body.mobileNumber,
									_id:req.params.id
								}
							);

							   if(req?.body?.isConfirmed !== null || req?.body?.isConfirmed != undefined)
							   {
                                  user.isConfirmed = req?.body?.isConfirmed 
							   }
								// update book.

								console.log(req.params.id)
								User.findByIdAndUpdate(req.params.id, user, {},function (err) {
									if (err) { 
										return apiResponse.ErrorResponse(res, err); 
									}else{
										let userData = new UserData(user);
										return apiResponse.successResponseWithData(res,"User update Success.", userData);
									}
								});
				
						}
					});
				}
			}
		} catch (err) {
			//throw error in json response with status 500. 
			return apiResponse.ErrorResponse(res, err);
		}
	}
];

/**
 * Book Delete.
 * 
 * @param {string}      id
 * 
 * @returns {Object}
 */
exports.userDelete = [
	auth,
	function (req, res) {
		if(!mongoose.Types.ObjectId.isValid(req.params.id)){
			return apiResponse.validationErrorWithData(res, "Invalid Error.", "Invalid ID");
		}
		try {
			User.findById(req.params.id, function (err, foundUser) {
				if(foundUser === null){
					return apiResponse.notFoundResponse(res,"User not exists with this id");
				}else{
					
						//delete book.
						User.findByIdAndRemove(req.params.id,function (err) {
							if (err) { 
								return apiResponse.ErrorResponse(res, err); 
							}else{
								return apiResponse.successResponse(res,"User delete Success.");
							}
						});
				
				}
			});
		} catch (err) {
			//throw error in json response with status 500. 
			return apiResponse.ErrorResponse(res, err);
		}
	}
];

exports.userBulkDelete = [
	auth,
	function (req, res) {
        if(!Array.isArray(req.body.users)){
            	return apiResponse.validationErrorWithData(res, "Invalid Error.", "Invalid ID");
        } 
        
        for(var i= 0 ; i < req.body.users.length ; i++)
        {
            if(!mongoose.Types.ObjectId.isValid(req.body.users[i])){
            	return apiResponse.validationErrorWithData(res, "Invalid Error.", "Invalid ID");
            } 
        }

		try {
            User.deleteMany({ _id : {$in : req.body.users}}).then(result => {
                return apiResponse.successResponse(res,"User delete Success.");
            })
            .catch(err => {
                return apiResponse.ErrorResponse(res, err);
            })
		} catch (err) {
			//throw error in json response with status 500. 
			return apiResponse.ErrorResponse(res, err);
		}
	}
];