const Language = require("../models/LanguageModel");
const { body,validationResult } = require("express-validator");
const { sanitizeBody } = require("express-validator");
const apiResponse = require("../helpers/apiResponse");
const auth = require("../middlewares/jwt");
var mongoose = require("mongoose");
mongoose.set("useFindAndModify", false);


// Book Schema
function LanguageData(data) {
	this.id = data._id;
	this.name = data.name;
    this.keyword = data.keyword
}

/**
 * Book List.
 * 
 * @returns {Object}
 */
exports.langList = [
	auth,
	function (req, res) {
		try {
			Language.find({},"_id name keyword createdAt").then((languages)=>{
				if(languages.length > 0){
					return apiResponse.successResponseWithData(res, "Operation success", languages);
				}else{
					return apiResponse.successResponseWithData(res, "Operation success", []);
				}
			});
		} catch (err) {
			//throw error in json response with status 500. 
			return apiResponse.ErrorResponse(res, err);
		}
	}
];

/**
 * Book Detail.
 * 
 * @param {string}      id
 * 
 * @returns {Object}
 */
exports.langDetail = [
	auth,
	function (req, res) {
		if(!mongoose.Types.ObjectId.isValid(req.params.id)){
			return apiResponse.successResponseWithData(res, "Operation success", {});
		}
		try {
			Language.findOne({_id: req.params.id},"_id name keyword createdAt").then((language)=>{                
				if(language !== null){
					let languageData = new LanguageData(language);
					return apiResponse.successResponseWithData(res, "Operation success", languageData);
				}else{
					return apiResponse.successResponseWithData(res, "Operation success", {});
				}
			});
		} catch (err) {
			//throw error in json response with status 500. 
			return apiResponse.ErrorResponse(res, err);
		}
	}
];


exports.langStore = [
	auth,
	body("name").isLength({ min: 1 }).trim().withMessage("Name must be specified.")
		.isLength({min : 2}).withMessage("Name must me minimum 2 characters long."),
    body("keyword").isLength({ min: 1 }).trim().withMessage("Keyword must be specified.")
		.isLength({min : 2}).withMessage("Keyword must me minimum 2 characters long."),
	sanitizeBody("name").escape(),
    sanitizeBody("keyword").escape(),
	(req, res) => {
		try {
			const errors = validationResult(req);
			if (!errors.isEmpty()) {
				return apiResponse.validationErrorWithData(res, "Validation Error.", errors.array());
			}
			else {

					// Create User object with escaped and trimmed data
					var language = new Language(
						{
							name: req.body.name,
                            keyword : req.body.keyword
						}
					);
						// Save user.
						language.save(function (err) {
							if (err) { return apiResponse.ErrorResponse(res, err); }
							let languageData = {
								_id: language._id,
								name: language.name,
                                keyword : language.keyword
							};
							return apiResponse.successResponseWithData(res,"Language Created.", languageData);
						});

			}
		} catch (err) {
			//throw error in json response with status 500. 
			return apiResponse.ErrorResponse(res, err);
		}
	}
];

/**
 * Book update.
 * 
 * @param {string}      title 
 * @param {string}      description
 * @param {string}      isbn
 * 
 * @returns {Object}
 */
exports.langUpdate = [
	auth,
	body("name").isLength({ min: 1 }).trim().withMessage("Name must be specified.")
		.isLength({min : 2}).withMessage("Name must be at least 2 characters long."),
	sanitizeBody("name").escape(),
	(req, res) => {
		try {
			const errors = validationResult(req);
			if (!errors.isEmpty()) {
				return apiResponse.validationErrorWithData(res, "Validation Error.", errors.array());
			}
			else {
				if(!mongoose.Types.ObjectId.isValid(req.params.id)){
					return apiResponse.validationErrorWithData(res, "Invalid Error.", "Invalid ID");
				}else{
					Language.findById(req.params.id, function (err, foundLang) {
						if(foundLang === null){
							return apiResponse.notFoundResponse(res,"Language not exists with this id");
						}else{

							var language = new Language(
								{
									name: req.body.name,
									_id:req.params.id,
                                    keyword : req.body.keyword
								}
							);
				
								Language.findByIdAndUpdate(req.params.id, language, {},function (err) {
									if (err) { 
										return apiResponse.ErrorResponse(res, err); 
									}else{
										let languageData = new LanguageData(language);
										return apiResponse.successResponseWithData(res,"Language update Success.", languageData);
									}
								});
				
						}
					});
				}
			}
		} catch (err) {
			//throw error in json response with status 500. 
			return apiResponse.ErrorResponse(res, err);
		}
	}
];

/**
 * Book Delete.
 * 
 * @param {string}      id
 * 
 * @returns {Object}
 */
exports.langDelete = [
	auth,
	function (req, res) {
		if(!mongoose.Types.ObjectId.isValid(req.params.id)){
			return apiResponse.validationErrorWithData(res, "Invalid Error.", "Invalid ID");
		}
		try {
			Language.findById(req.params.id, function (err, foundLang) {
				if(foundLang === null){
					return apiResponse.notFoundResponse(res,"Language not exists with this id");
				}else{
						//delete book.
						Language.findByIdAndRemove(req.params.id,function (err) {
							if (err) { 
								return apiResponse.ErrorResponse(res, err); 
							}else{
								return apiResponse.successResponse(res,"Language delete Success.");
							}
						});
				
				}
			});
		} catch (err) {
			//throw error in json response with status 500. 
			return apiResponse.ErrorResponse(res, err);
		}
	}
];


exports.langBulkDelete = [
	auth,
	function (req, res) {
		console.log(req.body.languages)
        if(!Array.isArray(req.body.languages)){
            	return apiResponse.validationErrorWithData(res, "Invalid Error.", "Invalid ID");
        } 
        
        for(var i= 0 ; i < req.body.languages.length ; i++)
        {
            if(!mongoose.Types.ObjectId.isValid(req.body.languages[i])){
            	return apiResponse.validationErrorWithData(res, "Invalid Error.", "Invalid ID");
            } 
        }
		try {
            Language.deleteMany({ _id : {$in : req.body.languages}}).then(result => {
                return apiResponse.successResponse(res,"Languages delete Success.");
            })
            .catch(err => {
                return apiResponse.ErrorResponse(res, err);
            })
		} catch (err) {
			//throw error in json response with status 500. 
			return apiResponse.ErrorResponse(res, err);
		}
	}
];