const LoanType = require("../models/LoanTypeModel");
const { body,validationResult } = require("express-validator");
const { sanitizeBody } = require("express-validator");
const apiResponse = require("../helpers/apiResponse");
const auth = require("../middlewares/jwt");
var mongoose = require("mongoose");
mongoose.set("useFindAndModify", false);
const { constants } = require("../helpers/constants");
const bcrypt = require("bcrypt");
const utility = require("../helpers/utility");
const multer = require('multer');
const path = require('path');

// Book Schema
function LoanTypeData(data) {
	this.id = data._id;
	this.name= data.name;
	this.maxAmount = data.maxAmount,
	this.image = data.image,
	this.imageUrl = data.imageUrl
}

/**
 * Book List.
 * 
 * @returns {Object}
 */
exports.loanTypeList = [
	auth,
	function (req, res) {
		try {
			LoanType.find({},"_id name maxAmount image imageUrl createdAt").then((loanTypes)=>{
				if(loanTypes.length > 0){
					return apiResponse.successResponseWithData(res, "Operation success", loanTypes);
				}else{
					return apiResponse.successResponseWithData(res, "No loan type found", []);
				}
			});
		} catch (err) {
			//throw error in json response with status 500. 
			return apiResponse.ErrorResponse(res, err);
		}
	}
];

/**
 * Book Detail.
 * 
 * @param {string}      id
 * 
 * @returns {Object}
 */
exports.loanTypeDetail = [
	auth,
	function (req, res) {
		if(!mongoose.Types.ObjectId.isValid(req.params.id)){
			return apiResponse.successResponseWithData(res, "Operation success", {});
		}
		try {
			LoanType.findOne({_id: req.params.id},"_id name maxAmount image imageUrl createdAt").then((loanType)=>{                
				if(loanType !== null){
					let loanTypeData = new LoanTypeData(loanType);
					return apiResponse.successResponseWithData(res, "Operation success", loanTypeData);
				}else{
					return apiResponse.successResponseWithData(res, "No loan type found", {});
				}
			});
		} catch (err) {
			//throw error in json response with status 500. 
			return apiResponse.ErrorResponse(res, err);
		}
	}
];

const storage = multer.diskStorage({
    destination: function(req, file, cb) {
        cb(null, 'uploads/');
    },

    // By default, multer removes file extensions so let's add them back
    filename: function(req, file, cb) {
		cb(null, Date.now() + '-' + file.originalname);
    }
});


exports.loanTypeStore = [
	auth,
	body("name").isLength({ min: 1 }).trim().withMessage("Name must be specified.")
		.isLength({min : 2}).withMessage("Name must me minimum 2 characters long."),
	body("maxAmount").isLength({ min: 1 }).trim().withMessage("Maximum Amount be specified.")
		.isNumeric().withMessage("Name must me minimum 2 characters long."),
	sanitizeBody("name").escape(),
	sanitizeBody("maxAmount").escape(),
	(req, res) => {
		try {
			const errors = validationResult(req);
			if (!errors.isEmpty()) {
				return apiResponse.validationErrorWithData(res, "Validation Error.", errors.array());
			}
			else {
				
				let data = req.body;
				let upload = multer({ storage: storage}).single('image');
				upload(req, res, function(err) {
					console.log('Here')
					if (req.fileValidationError) {
						return apiResponse.validationErrorWithData(res, "Validation Error.", req.fileValidationError)
					}
					else if (!req.file) {
						return res.status(400).send({ message: "Please select an image to upload" });
					}
					else if (err instanceof multer.MulterError) {
						return res.status(500).send({ message: `Could not upload the file: ${req.file.originalname}. ${err}` });
					}
					else if (err) {
						return res.send(err);
					}
	
					var baseUrl = req.protocol + "://" + req.headers.host;
					var loanType = new LoanType(
						{
						   name : data.name,
						   maxAmount : data.maxAmount,
						   image : req.file.filename,
						   imageUrl: baseUrl + `/uploads/` + req.file.filename
						}
					)
	
					loanType.save(function(err){
						if(err) { return apiResponse.ErrorResponse(res, err);}
						// return apiResponse.successResponseWithData(res,"Address Created.", address);
						let loanTypeData = {
							_id: loanType._id,
							name : loanType.name,
							maxAmount : loanType.maxAmount,
							image: loanType.image,
							imageUrl : loanType.imageUrl
						};
						return apiResponse.successResponseWithData(res,"Loan Type Created.", loanTypeData);
					})

					
				})

			}
		} catch (err) {
			//throw error in json response with status 500. 
			return apiResponse.ErrorResponse(res, err);
		}
	}
];

/**
 * Book update.
 * 
 * @param {string}      title 
 * @param {string}      description
 * @param {string}      isbn
 * 
 * @returns {Object}
 */
exports.loanTypeUpdate = [
	auth,
	body("name").isLength({ min: 1 }).trim().withMessage("Name must be specified.")
		.isLength({min : 2}).withMessage("Name must be at least 2 characters long."),
	body("maxAmount").isLength({ min: 1 }).trim().withMessage("Maximum Amount be specified.")
		.isNumeric().withMessage("Name must me minimum 2 characters long."),
	sanitizeBody("name").escape(),
	sanitizeBody("maxAmount").escape(),
	(req, res) => {
		try {
			const errors = validationResult(req);
			if (!errors.isEmpty()) {
				return apiResponse.validationErrorWithData(res, "Validation Error.", errors.array());
			}
			else {
				if(!mongoose.Types.ObjectId.isValid(req.params.id)){
					return apiResponse.validationErrorWithData(res, "Invalid Error.", "Invalid ID");
				}else{
					if(!mongoose.Types.ObjectId.isValid(req.params.id)){
						return apiResponse.validationErrorWithData(res, "Invalid Error.", "Invalid ID");
					}else{
						LoanType.findById(req.params.id, function (err, foundLoanType) {
							if(foundLoanType === null){
								return apiResponse.notFoundResponse(res,"Loan type exists with this id");
							}else{
	
								let data = req.body;
								let upload = multer({ storage: storage}).single('image');
								upload(req, res, function(err) {
									if (req.fileValidationError) {
										return apiResponse.validationErrorWithData(res, "Validation Error.", req.fileValidationError)
									}
									else if (!req.file) {
										return res.status(400).send({ status : 0, message: "Please select an image to upload" ,data : null});
									}
									else if (err instanceof multer.MulterError) {
										return res.status(500).send({ status : 0, message: `Could not upload the file: ${req.file.originalname}. ${err}`, data : null });
									}
									else if (err) {
										return apiResponse.ErrorResponse(res, err);
									}
	
									var baseUrl = req.protocol + "://" + req.headers.host;
									
									var loanType = new LoanType(
										{
										   _id:req.params.id,
										   name : data.name,
										   maxAmount : data.maxAmount,
										   image : req.file.filename,
										   imageUrl: baseUrl + `/uploads/` + req.file.filename
										}
									)
	
					
								LoanType.findByIdAndUpdate(req.params.id, loanType, {},function (err) {
									if (err) { 
										return apiResponse.ErrorResponse(res, err); 
									}else{
										LoanType.findById(req.params.id).then((data) => {
											return apiResponse.successResponseWithData(res,"Loan Type update Success.", data);
										});
									}
								});
							  });
					
							}
						});
					}
				}
			}
		} catch (err) {
			//throw error in json response with status 500. 
			return apiResponse.ErrorResponse(res, err);
		}
	}
];

/**
 * Book Delete.
 * 
 * @param {string}      id
 * 
 * @returns {Object}
 */
exports.loanTypeDelete = [
	auth,
	function (req, res) {
		if(!mongoose.Types.ObjectId.isValid(req.params.id)){
			return apiResponse.validationErrorWithData(res, "Invalid Error.", "Invalid ID");
		}
		try {
			LoanType.findById(req.params.id, function (err, foundLoanType) {
				if(foundLoanType === null){
					return apiResponse.notFoundResponse(res,"Loan type not exists with this id");
				}else{
						//delete book.
						LoanType.findByIdAndRemove(req.params.id,function (err) {
							if (err) { 
								return apiResponse.ErrorResponse(res, err); 
							}else{
								return apiResponse.successResponse(res,"Loan Type delete Success.");
							}
						});
				
				}
			});
		} catch (err) {
			//throw error in json response with status 500. 
			return apiResponse.ErrorResponse(res, err);
		}
	}
];


exports.loanTypeBulkDelete = [
	auth,
	function (req, res) {
		console.log(req.body.loan_types)
        if(!Array.isArray(req.body.loan_types)){
            	return apiResponse.validationErrorWithData(res, "Invalid Error.", "Invalid ID");
        } 
        
        for(var i= 0 ; i < req.body.loan_types.length ; i++)
        {
            if(!mongoose.Types.ObjectId.isValid(req.body.loan_types[i])){
            	return apiResponse.validationErrorWithData(res, "Invalid Error.", "Invalid ID");
            } 
        }
		try {
            LoanType.deleteMany({ _id : {$in : req.body.loan_types}}).then(result => {
                return apiResponse.successResponse(res,"Loan types delete Success.");
            })
            .catch(err => {
                return apiResponse.ErrorResponse(res, err);
            })
		} catch (err) {
			//throw error in json response with status 500. 
			return apiResponse.ErrorResponse(res, err);
		}
	}
];